package com.cogniance.acs.server;

import com.cogniance.acs.messaging.Fault;
import com.cogniance.acs.messaging.Inform;
import com.cogniance.acs.messaging.api.Message;
import com.cogniance.acs.messaging.api.helping.ACSMessage;
import com.cogniance.acs.messaging.api.helping.CPEMessage;
import com.cogniance.acs.server.api.AwaitTimeoutException;
import com.cogniance.acs.server.api.Session;
import com.cogniance.acs.util.exception.ACSRuntimeException;

import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.TimeUnit;

class SessionImpl implements Session {

    private final LinkedList<MessageInfo> messagesInfo = new LinkedList<MessageInfo>();

    private boolean holdRequests = false;
    private boolean cpeFinishedRequests = false;

    public SessionImpl() {
    }

    @Override
    public synchronized Inform getInform() {
        return (Inform) messagesInfo.getFirst().message;
    }

    @Override
    public synchronized Message getLastMessage() {
        return messagesInfo.getLast().message;
    }

    @Override
    public synchronized List<Message> getMessages() {
        List<Message> result = new ArrayList<Message>(this.messagesInfo.size());
        for (MessageInfo messageInfo : this.messagesInfo) {
            result.add(messageInfo.message);
        }
        return result;
    }

    @Override
    public void completeSession() throws InterruptedException {
        while (!isFinished()) {
            CPEMessage message;
            try {
                message = awaitCPEMessage(10, TimeUnit.MINUTES);
            } catch (AwaitTimeoutException e) {
                throw new ACSRuntimeException("Failed to complete session: no message from CPE.", e);
            }
            if (message == null) {
                sendACSMessage(null);
            } else {
                sendACSMessage((ACSMessage) message.createResponseMessage());
            }
        }
    }

    @Override
    public synchronized void sendACSMessage(ACSMessage acsMessage) {
        registerMessage(acsMessage, MessageOrigin.ACS);
        updateHoldRequests(acsMessage);
    }

    private void updateHoldRequests(ACSMessage acsMessage) {
        if (acsMessage != null && acsMessage.isHoldRequests()) {
            holdRequests = true;
        } else if (acsMessage == null || !acsMessage.isHoldRequests()) {
            holdRequests = false;
        }
    }

    public synchronized void registerCPEMessage(CPEMessage cpeMessage) {
        registerMessage(cpeMessage, MessageOrigin.CPE);
        updateCPEFinishedRequests(cpeMessage);
    }

    private void updateCPEFinishedRequests(CPEMessage cpeMessage) {
        if (!holdRequests && cpeMessage == null) {
            cpeFinishedRequests = true;
        }
    }

    private void registerMessage(Message message, MessageOrigin origin) {
        messagesInfo.add(new MessageInfo(message, origin));
        notifyAll();
    }

    @Override
    public synchronized <T extends CPEMessage> T awaitCPEMessage(long timeout, TimeUnit timeUnit) throws InterruptedException, AwaitTimeoutException {
        long deadline = System.nanoTime() + timeUnit.toNanos(timeout);
        while (messagesInfo.isEmpty() || messagesInfo.getLast().origin != MessageOrigin.CPE) {
            long millisToWait = TimeUnit.NANOSECONDS.toMillis(deadline - System.nanoTime());
            if (millisToWait <= 0) {
                break;
            }
            wait(millisToWait);
        }
        if (messagesInfo.isEmpty() || messagesInfo.getLast().origin != MessageOrigin.CPE) {
            throw new AwaitTimeoutException();
        }
        return (T) messagesInfo.getLast().message;
    }

    public synchronized ACSMessage awaitACSMessage(long timeout, TimeUnit timeUnit) throws InterruptedException, AwaitTimeoutException {
        long deadline = System.nanoTime() + timeUnit.toNanos(timeout);
        while (!(messagesInfo.getLast().origin == MessageOrigin.ACS)) {
            long millisToWait = TimeUnit.NANOSECONDS.toMillis(deadline - System.nanoTime());
            if (millisToWait <= 0) {
                break;
            }
            wait(millisToWait);
        }
        if (messagesInfo.getLast().origin == MessageOrigin.ACS) {
            return (ACSMessage) messagesInfo.getLast().message;
        }
        throw new AwaitTimeoutException();
    }

    @Override
    public synchronized boolean isFinished() {
        if (messagesInfo.isEmpty()) {
            return false;
        }
        if (messagesInfo.getLast().message instanceof Fault) {
            return true;
        }
        if (messagesInfo.size() % 2 == 0) { // even messages count
            return messagesInfo.getLast().message == null && cpeFinishedRequests;
        }
        return false;
    }

    @Override
    public synchronized boolean isCPEFinishedRequests() {
        return cpeFinishedRequests;
    }

    private static class MessageInfo {
        Message message;
        Date occurrenceTime;
        MessageOrigin origin;

        public MessageInfo(Message message, MessageOrigin origin) {
            this.message = message;
            this.occurrenceTime = new Date();
            this.origin = origin;
        }
    }

    private enum MessageOrigin {
        ACS, CPE
    }
}
