package com.cogniance.acs.server;

import com.cogniance.acs.util.exception.ACSRuntimeException;
import org.apache.commons.io.FileUtils;
import org.apache.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import static org.apache.commons.io.FileUtils.forceMkdir;

class FileServingServlet extends HttpServlet {

    private static final Logger logger = LoggerFactory.getLogger(FileServingServlet.class);

    private final Map<String, File> pathSpecToFile = new HashMap<String, File>();
    private final File filesUploadDirectory;

    public FileServingServlet(File filesUploadDirectory) {
        this.filesUploadDirectory = filesUploadDirectory;
        try {
            forceMkdir(filesUploadDirectory);
        } catch (IOException e) {
            throw new ACSRuntimeException("Failed to create directory for files upload.", e);
        }
    }

    void exposeFile(File originalFile, String pathSpec) {
        if (pathSpecToFile.containsKey(pathSpec)) {
            logger.warn("Overwriting path spec - file mapping for: {}", pathSpec);
        }
        pathSpecToFile.put(pathSpec, originalFile);
    }

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        String fileRelativePath = req.getPathInfo();
        File fileToGet = pathSpecToFile.get(fileRelativePath);
        if (fileToGet == null) {
            resp.sendError(HttpStatus.SC_NOT_FOUND);
            return;
        }
        resp.setStatus(HttpStatus.SC_OK);
        resp.setContentLength((int) fileToGet.length());
        resp.setContentType("application/octet-stream");
        FileUtils.copyFile(fileToGet, resp.getOutputStream());
        resp.getOutputStream().close();
    }
    // TODO authentication
    @Override
    protected void doPut(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {
        String fileRelativePath = req.getPathInfo();
        File fileToPut = new File(filesUploadDirectory, fileRelativePath);
        if (!isParentChild(filesUploadDirectory, fileToPut)) {
            resp.sendError(HttpStatus.SC_FORBIDDEN);
            return;
        }
        if (fileToPut.exists()) {
            resp.sendError(HttpStatus.SC_CONFLICT, String.format("File '%s' already exists, overwrites forbidden.", fileRelativePath));
            return;
        }
        FileUtils.forceMkdir(fileToPut.getParentFile());
        FileUtils.copyInputStreamToFile(req.getInputStream(), fileToPut);
        resp.setStatus(HttpStatus.SC_NO_CONTENT);
    }

    static private boolean isParentChild(File parent, File child) throws IOException {
        String parentPath = parent.getCanonicalPath();
        String childPath = child.getCanonicalPath();
        return childPath.startsWith(parentPath);
    }
}
