package com.cogniance.acs.server;

import com.cogniance.acs.server.api.AwaitTimeoutException;
import com.cogniance.acs.server.api.Device;
import com.cogniance.acs.server.api.Session;
import com.cogniance.acs.util.exception.ACSRuntimeException;
import org.apache.http.HttpResponse;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.DefaultConnectionReuseStrategy;
import org.apache.http.impl.client.DefaultHttpClient;
import org.eclipse.jetty.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

class DeviceImpl implements Device {

    private static final Logger logger = LoggerFactory.getLogger(DeviceImpl.class);

    private final List<String> protocolViolations = new ArrayList<String>();

    private final String serial;

    private String acsLogin;
    private String acsPassword;

    private String connectionRequestUrl;
    private String connectionRequestLogin;
    private String connectionRequestPassword;

    private final Object sessionMonitor = new Object();
    private SessionImpl currentSession;

    public DeviceImpl(String serial, String acsLogin, String acsPassword) {
        this.serial = serial;
        this.acsLogin = acsLogin;
        this.acsPassword = acsPassword;
    }

    @Override
    public String getSerial() {
        return serial;
    }

    public String getConnectionRequestUrl() {
        return connectionRequestUrl;
    }

    public void setConnectionRequestUrl(String connectionRequestUrl) {
        this.connectionRequestUrl = connectionRequestUrl;
    }

    @Override
    public String getConnectionRequestLogin() {
        return connectionRequestLogin;
    }

    @Override
    public void setConnectionRequestLogin(String connectionRequestLogin) {
        this.connectionRequestLogin = connectionRequestLogin;
    }

    @Override
    public String getConnectionRequestPassword() {
        return connectionRequestPassword;
    }

    @Override
    public void setConnectionRequestPassword(String connectionRequestPassword) {
        this.connectionRequestPassword = connectionRequestPassword;
    }

    @Override
    public String getACSLogin() {
        return acsLogin;
    }

    public void setACSLogin(String acsLogin) {
        this.acsLogin = acsLogin;
    }

    @Override
    public String getACSPassword() {
        return acsPassword;
    }

    public void setACSPassword(String acsPassword) {
        this.acsPassword = acsPassword;
    }

    @Override
    public boolean isInSession() {
        return currentSession != null;
    }

    @Override
    public SessionImpl getCurrentSession() {
        return currentSession;
    }

    @Override
    public Session awaitSession(long timeout, TimeUnit timeUnit) throws AwaitTimeoutException, InterruptedException {
        final long deadline = System.nanoTime() + timeUnit.toNanos(timeout);
        synchronized (sessionMonitor) {
            while (currentSession == null) {
                long millisToWait = TimeUnit.NANOSECONDS.toMillis(deadline - System.nanoTime());
                if (millisToWait <= 0) {
                    break;
                }
                sessionMonitor.wait(millisToWait);
            }
            if (currentSession != null) {
                return currentSession;
            }
            throw new AwaitTimeoutException();
        }
    }

    @Override
    public void awaitSessionFinished(long timeout, TimeUnit timeUnit) throws AwaitTimeoutException, InterruptedException {
        final long deadline = System.nanoTime() + timeUnit.toNanos(timeout);
        synchronized (sessionMonitor) {
            while (currentSession != null) {
                long millisToWait = TimeUnit.NANOSECONDS.toMillis(deadline - System.nanoTime());
                if (millisToWait <= 0) {
                    break;
                }
                sessionMonitor.wait(millisToWait);
            }
            if (currentSession == null) {
                return;
            }
            throw new AwaitTimeoutException();
        }
    }

    @Override
    public Session initiateSession(long timeout, TimeUnit timeUnit) throws InterruptedException, AwaitTimeoutException {
        final long deadline = System.nanoTime() + timeUnit.toNanos(timeout);
        if (currentSession != null) {
            throw new IllegalStateException("Session already in progress, can't initiate another one.");
        }
        sendConnectionRequest();
        return awaitSession(deadline - System.nanoTime(), TimeUnit.NANOSECONDS);
    }

    @Override
    public void sendConnectionRequest() {
        logger.info("Sending connection request to {} at {} user: {}", serial, connectionRequestUrl, connectionRequestLogin);
        DefaultHttpClient httpClient = new DefaultHttpClient();
        httpClient.getCredentialsProvider().setCredentials(
                AuthScope.ANY,
                new UsernamePasswordCredentials(connectionRequestLogin, connectionRequestPassword)
        );
        httpClient.setReuseStrategy(new DefaultConnectionReuseStrategy());
        HttpGet request = new HttpGet(connectionRequestUrl);
        try {
            HttpResponse response = httpClient.execute(request);
            if (!HttpStatus.isSuccess(response.getStatusLine().getStatusCode())) {
                logger.warn("Failed to send connection request: {}", response.getStatusLine());
                throw new ACSRuntimeException("Connection request rejected: %s", response.getStatusLine());
            }
        } catch (IOException e) {
            logger.warn("Failed to send connection request.", e);
            throw new ACSRuntimeException("Failed to send connection request to the CPE at %s.", connectionRequestUrl, e);
        }
    }

    public SessionImpl initializeSession() {
        synchronized (sessionMonitor) {
            currentSession = new SessionImpl();
            sessionMonitor.notifyAll();
            return currentSession;
        }
    }

    public void closeSession() {
        synchronized (sessionMonitor) {
            currentSession = null;
            sessionMonitor.notifyAll();
        }
    }

    public void registerProtocolViolation(String reason) {
        logger.error("Protocol violation happened.\nDevice: {}\nReason: {}", this, reason);
        synchronized (protocolViolations) {
            protocolViolations.add(reason);
        }
    }

    @Override
    public List<String> listProtocolViolations() {
        synchronized (protocolViolations) {
            return new ArrayList<String>(protocolViolations);
        }
    }
}
