package com.cogniance.acs.messaging.params;

import com.cogniance.acs.messaging.util.DataType;
import com.cogniance.acs.util.soap.SOAPUtils;
import com.google.common.base.Objects;

import javax.xml.soap.Name;
import javax.xml.soap.SOAPElement;

import static com.cogniance.acs.util.soap.SOAPUtils.getChildElement;
import static com.google.common.base.Objects.equal;
import static com.google.common.base.Objects.toStringHelper;

/**
 * Holds name and value of a parameter. Type is of "anySimpleType", so it can be "xsd:string", "xsd:int" and so on.
 */
public class ParameterValueStruct implements Comparable<ParameterValueStruct> {
    private String name;
    private String value;
    private String type;

    public ParameterValueStruct(SOAPElement element) {
        SOAPElement valueElement = getChildElement(element, "Value");
        SOAPElement nameElement = getChildElement(element, "Name");
        this.name = nameElement.getValue();
        this.type = normalizeType(valueElement.getAttribute("xsi:type"));
        this.value = valueElement.getValue();
    }

    public ParameterValueStruct(String name, String type, String value) {
        this.name = name;
        this.value = value;
        this.type = normalizeType(type);
    }

    public String getName() {
        return name;
    }

    public String getValue() {
        return value;
    }

    public String getType() {
        return type;
    }

    public DataType getDataType() {
        return DataType.fromValueType(type);
    }

    public Object getValueObject() {
        return getDataType().parse(value);
    }

    public int compareTo(ParameterValueStruct o) {
        if (this.name.compareTo(o.getName()) == 0 && this.value.compareTo(o.getValue()) == 0 && this.type.compareTo(o.getType()) == 0) {
            return 0;
        } else {
            return 1;
        }
    }

    @Override
    public int hashCode() {
        return Objects.hashCode(name, type, getValueObject());
    }

    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof ParameterValueStruct)) {
            return false;
        }
        ParameterValueStruct other = (ParameterValueStruct) obj;
        return equal(name, other.name) &&
                equal(type, other.type) &&
                equal(getValueObject(), other.getValueObject());
    }

    @Override
    public String toString() {
        return toStringHelper(this)
                .add("name", name)
                .add("type", type)
                .add("value", value)
                .toString();
    }

    private static String normalizeType(String type) {
        return type.startsWith("xsd:") ? type : "xsd:" + type;
    }

    public static <T> ParameterValueStruct pvsOf(String name, DataType<T> dataType, T value) {
        return of(name, dataType, value);
    }

    public static <T> ParameterValueStruct of(String name, DataType<T> dataType, T value) {
        return new ParameterValueStruct(name, dataType.getValueType(), dataType.format(value));
    }
}
