package com.cogniance.acs.server;

import com.cogniance.acs.util.exception.ACSRuntimeException;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import org.eclipse.jetty.security.ConstraintMapping;
import org.eclipse.jetty.security.ConstraintSecurityHandler;
import org.eclipse.jetty.security.HashLoginService;
import org.eclipse.jetty.security.SecurityHandler;
import org.eclipse.jetty.security.authentication.BasicAuthenticator;
import org.eclipse.jetty.security.authentication.DigestAuthenticator;
import org.eclipse.jetty.util.security.Constraint;
import org.eclipse.jetty.util.security.Credential;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

public class ACSSecurityManager {
    private final HashLoginService loginService = new HashLoginService();
    private final ConstraintSecurityHandler securityHandler = new ConstraintSecurityHandler();

    private final Multimap<String, String> pathSpecToRoles = HashMultimap.create();

    public ACSSecurityManager(String authType) {
        if (Constraint.__BASIC_AUTH.equals(authType)) {
            securityHandler.setAuthenticator(new BasicAuthenticator());
        } else if (Constraint.__DIGEST_AUTH.equals(authType)) {
            securityHandler.setAuthenticator(new DigestAuthenticator());
        } else {
            throw new ACSRuntimeException("Passed auth type (%s) is not supported.", authType);
        }
        securityHandler.setLoginService(loginService);
    }

    public SecurityHandler getWebSecurityHandler() {
        return securityHandler;
    }

    public void addUser(String login, String password, String... roles) {
        if (roles == null) {
            return;
        }
        Credential credential = Credential.getCredential(password);
        loginService.putUser(login, credential, roles);
    }

    public void removeUser(String login) {
        loginService.removeUser(login);
    }

    public void setPathAllowedRoles(String pathSpec, Collection<String> allowedRoles) {
        synchronized (pathSpecToRoles) {
            pathSpecToRoles.removeAll(pathSpec);
            pathSpecToRoles.putAll(pathSpec, allowedRoles);
        }

        updateSecurityRules();
    }

    public void allowPathForRole(String pathSpec, String role) {
        synchronized (pathSpecToRoles) {
            pathSpecToRoles.put(pathSpec, role);
        }

        updateSecurityRules();
    }

    public void forbidPathForRole(String pathSpec, String role) {
        synchronized (pathSpecToRoles) {
            pathSpecToRoles.remove(pathSpec, role);
        }

        updateSecurityRules();
    }

    private void updateSecurityRules() {
        synchronized (pathSpecToRoles) {
            List<ConstraintMapping> mappings = new ArrayList<ConstraintMapping>(pathSpecToRoles.keySet().size());
            for (String pathSpec : pathSpecToRoles.keySet()) {
                Constraint constraint = new Constraint();
                constraint.setAuthenticate(true);
                constraint.setName(pathSpec);
                Collection<String> roles = pathSpecToRoles.get(pathSpec);
                String[] rolesArray = roles.toArray(new String[roles.size()]);
                constraint.setRoles(rolesArray);

                ConstraintMapping mapping = new ConstraintMapping();
                mapping.setConstraint(constraint);
                mapping.setPathSpec(pathSpec);

                mappings.add(mapping);
            }
            securityHandler.setConstraintMappings(mappings);
        }
    }


}
