package com.cogniance.acs.server;

import com.cogniance.acs.server.api.ACS;
import com.cogniance.acs.server.api.ACSConfig;
import com.cogniance.acs.server.api.Device;
import com.cogniance.acs.server.api.FileDownloadData;

import javax.servlet.http.HttpServlet;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

public class ACSImpl implements ACS {

    private ACSConfig config;

    private final Map<String, DeviceImpl> serialToDevice = new HashMap<String, DeviceImpl>();

    private ACSSecurityManager securityManager;

    private FileServingServlet fileServingServlet;
    private CWMPServlet cwmpServlet;

    private WebServer webServer;

    @Override
    public void start(ACSConfig config) {
        this.config = config;
        initializeSecurityManager();
        Map<HttpServlet, String> servlets = createServlets();

        webServer = new WebServer(config.getHost(), config.getPort());
        webServer.setupSecurityManager(securityManager);
        webServer.start(servlets);
    }

    private void initializeSecurityManager() {
        securityManager = new ACSSecurityManager(config.getAuthType());
        securityManager.allowPathForRole(config.getCwmpPostfix(), "admin");
        securityManager.allowPathForRole(config.getFileServingPathSpec(), "admin");
    }

    private Map<HttpServlet, String> createServlets() {
        fileServingServlet = new FileServingServlet(config.getFileServingDirectory());
        cwmpServlet = new CWMPServlet(this);

        Map<HttpServlet, String> servletToPathSpec = new HashMap<HttpServlet, String>();
        servletToPathSpec.put(cwmpServlet, config.getCwmpPostfix());
        servletToPathSpec.put(fileServingServlet, config.getFileServingPathSpec());
        return servletToPathSpec;
    }

    @Override
    public void stop() {
        webServer.stop();
    }

    @Override
    public DeviceImpl registerDevice(String serial, String acsLogin, String acsPassword) {
        DeviceImpl device = new DeviceImpl(serial, acsLogin, acsPassword);
        if (serialToDevice.containsKey(device.getSerial())) {
            throw new IllegalStateException("Device with such serial already exists: " + device.getSerial());
        }
        serialToDevice.put(device.getSerial(), device);
        securityManager.addUser(device.getACSLogin(), device.getACSPassword(), device.getSerial());
        securityManager.allowPathForRole(config.getCwmpPostfix(), device.getSerial());
        return device;
    }

    @Override
    public void setDeviceCredentials(Device device, String acsLogin, String acsPassword) {
        securityManager.removeUser(device.getACSLogin());
        DeviceImpl realDevice = serialToDevice.get(device.getSerial());
        realDevice.setACSLogin(acsLogin);
        realDevice.setACSPassword(acsPassword);
        securityManager.addUser(device.getACSLogin(), device.getACSPassword(), device.getSerial());
    }

    @Override
    public DeviceImpl getDevice(String serial) {
        return serialToDevice.get(serial);
    }

    @Override
    public void unregisterDevice(Device device) {
        serialToDevice.remove(device.getSerial());
        securityManager.forbidPathForRole(config.getCwmpPostfix(), device.getSerial());
        securityManager.removeUser(device.getACSLogin());
    }

    @Override
    public FileDownloadData exposeFileForDownload(File realFile, String pathSpec) throws IOException {
        fileServingServlet.exposeFile(realFile, pathSpec);
        URL url = new URL("http", config.getHost(), config.getPort(), config.getFileServingPostfix() + pathSpec);
        String role = "download:" + pathSpec;
        String login = UUID.randomUUID().toString();
        String password = UUID.randomUUID().toString();
        securityManager.allowPathForRole(url.getFile(), role);
        securityManager.addUser(login, password, role);
        return new FileDownloadData(url, login, password); // TODO roles clean-up
    }
}
