package com.cogniance.acs.messaging;

import com.cogniance.acs.messaging.api.CPERequest;
import com.cogniance.acs.messaging.params.DeviceId;
import com.cogniance.acs.messaging.params.EventStruct;
import com.cogniance.acs.messaging.params.ParameterValueStruct;

import javax.xml.soap.Name;
import javax.xml.soap.SOAPBodyElement;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPException;
import java.util.ArrayList;
import java.util.Iterator;

import static com.cogniance.acs.util.soap.SOAPUtils.getChildElement;
import static com.cogniance.acs.util.soap.SOAPUtils.getChildElementValue;
import static com.google.common.base.Objects.equal;

/**
 * This method is used for a CPE to initiate a transaction, directly after a session is established.
 */
public class Inform extends AbstractMessage implements CPERequest {
    private DeviceId deviceId;
    private ArrayList<EventStruct> events;
    private Integer maxEnvelopes;
    private String currentTime;
    private Integer retryCount;
    private ArrayList<ParameterValueStruct> parameterList;

    public Inform() {
        super("Inform");
    }

    @Override
    protected void parseSOAPBody(SOAPBodyElement body) throws SOAPException {
        this.deviceId = DeviceId.from(getChildElement(body, "DeviceId"));
        this.maxEnvelopes = Integer.parseInt(getChildElementValue(body, "MaxEnvelopes"));
        this.retryCount = Integer.parseInt(getChildElementValue(body, "RetryCount"));
        this.currentTime = getChildElementValue(body, "RetryCount");

        parseSOAPParameterList(getChildElement(body, "ParameterList"));
        parseSOAPEvents(getChildElement(body, "Event"));
    }

    private void parseSOAPEvents(SOAPElement eventsElement) throws SOAPException {
        this.events = new ArrayList<EventStruct>();

        Iterator eventsIt = eventsElement.getChildElements(SOAP_FACTORY.createName("EventStruct"));
        Name eCode = SOAP_FACTORY.createName("EventCode");
        Name eCmdKey = SOAP_FACTORY.createName("CommandKey");
        while (eventsIt.hasNext()) {
            SOAPElement event = (SOAPElement) eventsIt.next();
            String code = getChildElementValue(event, eCode);
            String cmdkey = getChildElementValue(event, eCmdKey);

            if (cmdkey == null)
                cmdkey = "";
            this.events.add(new EventStruct(code, cmdkey));
        }
    }

    private void parseSOAPParameterList(SOAPElement parameterListElement) throws SOAPException {
        this.parameterList = new ArrayList<ParameterValueStruct>();

        Iterator paramListIt = parameterListElement.getChildElements(SOAP_FACTORY.createName("ParameterValueStruct"));
        while (paramListIt.hasNext()) {
            SOAPElement prm = (SOAPElement) paramListIt.next();
            this.parameterList.add(new ParameterValueStruct(prm));
        }
    }

    public DeviceId getDeviceId() {
        return deviceId;
    }

    public ArrayList<EventStruct> getEvents() {
        return events;
    }

    public Integer getMaxEnvelopes() {
        return maxEnvelopes;
    }

    public String getCurrentTime() {
        return currentTime;
    }

    public Integer getRetryCount() {
        return retryCount;
    }

    public ArrayList<ParameterValueStruct> getParameterList() {
        return parameterList;
    }

    public String getParameterValue(String parameterName) {
        for (ParameterValueStruct pvs : parameterList) {
            if (equal(pvs.getName(), parameterName)) {
                return pvs.getValue();
            }
        }
        return null;
    }

    public boolean containsEvent(String eventCode) {
        for (EventStruct eventStruct : events) {
            if (equal(eventStruct.getEvent(), eventCode)) {
                return true;
            }
        }
        return false;
    }

    public boolean containsEvent(String eventCode, String commandKey) {
        for (EventStruct eventStruct : events) {
            if (equal(eventStruct.getEvent(), eventCode) && equal(eventStruct.getCommandKey(), commandKey)) {
                return true;
            }
        }
        return false;
    }
}