package com.cogniance.acs.server;

import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.bio.SocketConnector;
import org.eclipse.jetty.server.handler.HandlerList;
import org.eclipse.jetty.server.session.SessionHandler;
import org.eclipse.jetty.servlet.ServletContextHandler;
import org.eclipse.jetty.servlet.ServletHolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServlet;
import java.util.Map;

class WebServer {

    private static final Logger logger = LoggerFactory.getLogger(WebServer.class);

    private final String host;
    private final int port;

    private ACSSecurityManager acsSecurityManager;

    private Server server;

    public WebServer(String host, int port) {
        this.host = host;
        this.port = port;
    }

    public void setupSecurityManager(ACSSecurityManager securityManager) {
        this.acsSecurityManager = securityManager;
    }

    public void start(Map<HttpServlet, String> servletToPathSpec) {
        server = new Server();

        initConnectors();
        initServlets(servletToPathSpec);

        try {
            server.start();
        } catch (Exception e) {
            logger.error("Failed to start web server.", e);
            throw new RuntimeException(e);
        }
    }

    private void initConnectors() {
        SocketConnector httpConnector = new SocketConnector();
        httpConnector.setHost(host);
        httpConnector.setPort(port);
        httpConnector.setMaxIdleTime(30000);

        server.addConnector(httpConnector);
    }

    private void initServlets(Map<HttpServlet, String> servletToPathSpec) {
        HandlerList rootHandlerList = new HandlerList();
        ServletContextHandler sch = new ServletContextHandler();
        sch.setSessionHandler(new SessionHandler());
        sch.setSecurityHandler(acsSecurityManager.getWebSecurityHandler());
        for (Map.Entry<HttpServlet, String> entry : servletToPathSpec.entrySet()) {
            HttpServlet servlet = entry.getKey();
            String pathSpec = entry.getValue();
            sch.addServlet(new ServletHolder(servlet), normalizePathSpec(pathSpec));
        }
        rootHandlerList.addHandler(sch);

        server.setHandler(rootHandlerList);
    }

    public void stop() {
        try {
            server.stop();
        } catch (Exception e) {
            logger.error("Failed to stop web server.", e);
        }
    }

    private static String normalizePathSpec(String pathSpec) {
        return pathSpec.replaceAll("/*$", "");
    }
}
