package com.cogniance.acs;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import stargate.util.crypto.DataEncryptor;

import com.cogniance.acs.messaging.Download;
import com.cogniance.acs.messaging.DownloadResponse;
import com.cogniance.acs.messaging.Fault;
import com.cogniance.acs.messaging.GetParameterValues;
import com.cogniance.acs.messaging.Inform;
import com.cogniance.acs.messaging.InformResponse;
import com.cogniance.acs.messaging.Reboot;
import com.cogniance.acs.messaging.SetParameterValues;
import com.cogniance.acs.messaging.TransferComplete;
import com.cogniance.acs.messaging.api.Message;
import com.cogniance.acs.messaging.api.helping.ACSMessage;
import com.cogniance.acs.messaging.api.helping.CPEMessage;
import com.cogniance.acs.messaging.params.ParameterValueStruct;
import com.cogniance.acs.server.ACSImpl;
import com.cogniance.acs.server.api.ACS;
import com.cogniance.acs.server.api.ACSConfig;
import com.cogniance.acs.server.api.AwaitTimeoutException;
import com.cogniance.acs.server.api.Device;
import com.cogniance.acs.server.api.FileDownloadData;
import com.cogniance.acs.server.api.Session;
import com.cogniance.acs.util.Utils;

public class Executor {
	private static final Logger logger = LoggerFactory.getLogger(Executor.class);
	private Config config;
	private Map<String, List<ACSMessage>> combos;

	private ACS acs;
	private Device device;

	public Executor(Config config) {
		this.config = config;
		combos = compileCombos();
	}
	
	private Session session;

	public void run(String firmwarePath) throws Exception {
		acs = new ACSImpl();

		startACS();
		try {
			prepareDevice();
			if (!Utils.isBlank(firmwarePath)) {
				initiateFirmwareInstall(firmwarePath);
				finishFirmwareInstall();
			} else {
				String cmdLine;
				Scanner scanner = new Scanner(System.in);
				while (true) {
					logger.info("Enter command...");
					cmdLine = scanner.nextLine();
					
					List<ACSMessage> messages = buildMessage(cmdLine);
					if (!messages.isEmpty()) {
						if (session == null || session.isFinished()) {
							session = initiateSession();
						}
						CPEMessage cpeMessage = session.awaitCPEMessage(10, TimeUnit.SECONDS);
						if (cpeMessage != null) {
							session.completeSession();
							device.awaitSessionFinished(5, TimeUnit.SECONDS);
							session = initiateSession();
						}
						for (ACSMessage message : messages) {
							session.sendACSMessage(message);
							CPEMessage response = session.awaitCPEMessage(1, TimeUnit.MINUTES);
						}
						session.completeSession();
					}
				}
			}
		} catch (AwaitTimeoutException ex) {
			logger.warn("Got timeout");
		} finally {
			stopACS();
		}
	}

	private void startACS() {
		ACSConfig acsConfig = new ACSConfig();
		acsConfig.setHost(config.getAcsHost()); // your host IP-address, visible
												// from CPE
		acsConfig.setPort(config.getAcsPort()); // any port you like
		acsConfig.setAuthType(config.getAuthType()); // auth type, according to the spec.
											// should be DIGEST
		acsConfig.setCwmpPostfix("/cpeManagement"); // should be same as in
													// CPE's config file
		acsConfig.setFileServingPostfix("/files"); // doesn't really affect
													// anything
		acsConfig.setFileServingDirectory(new File("/tmp/123/"));

		acs.start(acsConfig);
	}

	private void stopACS() {
		acs.stop();
	}

	private void prepareDevice() {
		// device = acs.registerDevice(DEVICE_SERIAL,
		// ACS_LOGIN,ACS_UNENCRYPTED_PASSWORD);
		device = acs.registerDevice(config.getDeviceSerial(), config.getAcsLogin(), getAcsPassword());
		device.setConnectionRequestUrl("http://" + config.getDeviceIp() + ":7547/");
		// our defaults
		device.setConnectionRequestLogin("kduam7W7FkUWKfDZ");
		device.setConnectionRequestPassword("ukzCvRYZKQzuhS9h");
	}

	private String getAcsPassword() {
		String password = null;
		if (!Utils.isBlank(config.getAcsUnencryptedPassword())) {
			password = config.getAcsUnencryptedPassword();
		} else {
			if (!Utils.isBlank(config.getAcsEncryptedPassword())) {
				DataEncryptor encryptor = new DataEncryptor();
				password = encryptor.decrypt(config.getAcsEncryptedPassword(), DataEncryptor.DEFAULT_KEY);
			}
		}

		if (password == null) {
			logger.error("ACS password is not specified");
		}

		return password;
	}

	private void initiateFirmwareInstall(String firmwarePath) throws InterruptedException, AwaitTimeoutException, IOException {
		Session session = initiateSession();

		for (CPEMessage cpeMessage = session.awaitCPEMessage(10, TimeUnit.SECONDS); cpeMessage != null; cpeMessage = session
				.awaitCPEMessage(10, TimeUnit.SECONDS)) {
			session.sendACSMessage((ACSMessage) cpeMessage.createResponseMessage());
		}

		Download downloadRequest = prepareDownload(firmwarePath);
		session.sendACSMessage(downloadRequest);

		CPEMessage response = session.awaitCPEMessage(1, TimeUnit.MINUTES);
		if (response instanceof Fault) {
			throw new IllegalStateException(((Fault) response).getCwmpFaultString());
		} else if (!(response instanceof DownloadResponse)) {
			throw new IllegalStateException("Invalid response: " + response);
		}
		session.completeSession();
		device.awaitSessionFinished(10, TimeUnit.SECONDS);
	}

	private Session initiateSession() throws InterruptedException, AwaitTimeoutException {
		Session session = device.initiateSession(10, TimeUnit.SECONDS);

		Inform inform = session.awaitCPEMessage(10, TimeUnit.SECONDS);
		session.sendACSMessage(new InformResponse());
		return session;
	}

	private Download prepareDownload(String firmwarePath) throws IOException {
		FileDownloadData data = acs.exposeFileForDownload(new File(firmwarePath), "/data.img.acs");

		Download downloadRequest = new Download();
		downloadRequest.setFileType(Download.FILE_TYPE_FIRMWARE_IMAGE);
		downloadRequest.setUsername(data.getUserName());
		downloadRequest.setPassword(data.getPassword());
		downloadRequest.setUrl(data.getFileUrl().toExternalForm());
		return downloadRequest;
	}

	private void finishFirmwareInstall() throws InterruptedException, AwaitTimeoutException {
		Session session = device.awaitSession(5, TimeUnit.MINUTES);
		session.completeSession();
		for (Message message : session.getMessages()) {
			if (message instanceof TransferComplete) {
				Integer faultCode = ((TransferComplete) message).getFaultStruct().getFaultCode();
				String faultString = ((TransferComplete) message).getFaultStruct().getFaultString();
				if (faultCode == 0) {
					return;
				} else {
					throw new IllegalStateException("Firmware install failed: " + faultCode + ", " + faultString);
				}
			}
		}
		throw new IllegalStateException("No firmware install occurred.");
	}

	private List<ACSMessage> buildMessage(String cmd) throws IOException {
		List<ACSMessage> messages = new ArrayList<ACSMessage>();
		Map<String, String> params = parseCommandLine(cmd);
		if (!validateCmd(params)) {
			return messages;
		}
		
		if (params.get("cmd").equals("get")) {
			GetParameterValues values = new GetParameterValues();
			values.addParameter(params.get("name"));
			messages.add(values);
		} else if (params.get("cmd").equals("set")) {
			SetParameterValues values = new SetParameterValues();
			values.addParameter(new ParameterValueStruct(params.get("name"), params.get("type"), params.get("value")));
			messages.add(values);
		} else if (params.get("cmd").equals("reboot")) {
			Reboot reboot = new Reboot();
			reboot.setCommandKey("");
			messages.add(reboot);
		}
		if (params.get("cmd").equals("combo")) {
			if (!combos.containsKey(params.get("name"))) {
				logger.warn("Not found combo for name=" + params.get("name"));
			} else {
				messages.addAll(combos.get(params.get("name")));
			}
		}

		if (params.get("cmd").equals("upgrade")) {
			FileDownloadData data = acs.exposeFileForDownload(new File(params.get("name")), "/data.img.acs");

			Download downloadRequest = new Download();
			downloadRequest.setFileType(Download.FILE_TYPE_FIRMWARE_IMAGE);
			downloadRequest.setUsername(data.getUserName());
			downloadRequest.setPassword(data.getPassword());
			downloadRequest.setUrl(data.getFileUrl().toExternalForm());

			messages.add(downloadRequest);
		}

		return messages;
	}

	private Map<String, List<ACSMessage>> compileCombos() {
		Map<String, List<ACSMessage>> combos = new HashMap<String, List<ACSMessage>>();
		for (Combo combo : config.getCombos()) {
			if (!combo.isValid()) {
				logger.error("Combo " + combo.getName() + " is not valid");
				throw new ComboException();
			}
			combos.put(combo.getName(), compileCombo(combo));
		}

		return combos;
	}

	private List<ACSMessage> compileCombo(Combo combo) {
		List<ACSMessage> messages = new ArrayList<ACSMessage>();
		String currentCmd = null;
		ACSMessage currentMsg = null;
		for (Command command : combo.getCommands()) {

			if (!command.getCmd().equals(currentCmd)) {
				if (command.getCmd().equals("get")) {
					GetParameterValues message = new GetParameterValues();
					messages.add(message);
					currentCmd = "get";
					currentMsg = message;
				} else {
					SetParameterValues message = new SetParameterValues();
					messages.add(message);
					currentCmd = "set";
					currentMsg = message;
				}
			}

			if (command.getCmd().equals("get")) {
				((GetParameterValues) currentMsg).addParameter(command.getName());
			} else {
				ParameterValueStruct value = new ParameterValueStruct(command.getName(), command.getType(), command.getValue());
				((SetParameterValues) currentMsg).addParameter(value);
			}
		}

		return messages;

	}

//	private Map<String, String> parseCommandLine(String cmd) {
//		Map<String, String> params = new HashMap<String, String>();
//		if (Utils.isBlank(cmd)) {
//			return params;
//		}
//
//		String[] pairs = cmd.split(" ");
//		for (String pair : pairs) {
//			if (Utils.isBlank(pair)) {
//				continue;
//			}
//			if (!pair.contains("=")) {
//				logger.warn("Invalid parameter found: " + pair);
//				continue;
//			}
//			String[] arr = pair.split("=");
//			params.put(arr[0], arr[1]);
//		}
//
//		return params;
//	}
	
	private Map<String, String> parseCommandLine(String cmd) {
		Map<String, String> params = new HashMap<String, String>();
		if (Utils.isBlank(cmd)) {
			return params;
		}

		String[] pairs = cmd.split(" ");
		for (String pair : pairs) {
			if (Utils.isBlank(pair)) {
				continue;
			}
			if (!pair.contains("=")) {
				logger.warn("Invalid parameter found: " + pair);
				continue;
			}
			String[] arr = pair.split("=");
			if (!arr[0].equals("value")) {
				params.put(arr[0], arr[1]);
			} else {
				params.put("value", cmd.substring(cmd.indexOf("value=") + 6));
				break;
			}
		}

		return params;
	}
	
	private boolean validateCmd(Map<String, String> params) {
		if (!params.containsKey("cmd")) {
			logger.warn("Missing 'cmd' parameter...");
			return false;
		}
		
		String cmd = params.get("cmd");
		if (isNameRequired(cmd) && !params.containsKey("name")) {
			logger.warn("Missing 'name' parameter...");
			return false;
		}


		if (cmd.equals("set")) {
			if (!params.containsKey("type")) {
				logger.warn("Missing 'type' parameter...");
				return false;
			}

			if (!params.containsKey("value")) {
				logger.warn("Missing 'value' parameter...");
				return false;
			}
		}

		return true;
	}
	
	private boolean isNameRequired(String cmd) {
		String[] commands = {"get", "set", "combo"};
		for (String command : commands) {
			if (command.equals(cmd)) {
				return true;
			}
		}
		
		return false;
	}
	
	
}
