// ac66u_updater.cpp : Defines the entry point for the console application.
//

#include "stdafx.h"

#include <boost/network/protocol/http.hpp>
#include <boost/network/uri.hpp>
#include <boost/archive/iterators/base64_from_binary.hpp>
#include <boost/archive/iterators/transform_width.hpp>
#include <boost/archive/iterators/ostream_iterator.hpp>
#include <boost/asio.hpp>
#include <boost/regex.hpp>
#include <boost/random/mersenne_twister.hpp>
#include <boost/random/uniform_int_distribution.hpp>
#include <boost/program_options.hpp>
#include <boost/process.hpp>
#include <boost/process/search_path.hpp>
#include <boost/assign/list_of.hpp>
#include <boost/algorithm/hex.hpp>
#include <cstdlib>
#include <cstdio>
#include <iostream>
#include <string>
#include <sstream>
#include <algorithm>
#include <fstream>
#include <map>
#include <vector>
#include <iterator>
#include <openssl/md5.h>

extern "C" { FILE __iob_func[3] = { *stdin,*stdout,*stderr }; }

class conf {
public:
	typedef boost::network::http::client			http_client;
	typedef boost::network::http::client::request	http_request;
	typedef boost::network::http::client::response	http_response;
	typedef boost::archive::iterators::base64_from_binary<
		boost::archive::iterators::transform_width<const char*, 6, 8> > base64_text;

private:
	http_client c;
	std::string username;
	std::string password;
	std::string auth_string;

	conf() {
	}
public:
	static conf& get() {
		static conf get;
		return get;
	}
	http_client& client() {
		return c;
	}
	void set_auth(const std::string& u, const std::string& p) {
		username = u;
		password = p;
		std::stringstream os;
		std::string auth = u + ":" + p;
		std::copy(conf::base64_text(auth.c_str()), conf::base64_text(auth.c_str() + auth.size()),
			boost::archive::iterators::ostream_iterator<char>(os));
		auth_string = std::string("Basic ") + os.str();
	}
	const std::string& auth_header() const {
		return auth_string;
	}
	const std::string& auth_username() const {
		return username;
	}
	const std::string& auth_password() const {
		return password;
	}
};

struct buffer {
	char* data;
	const size_t size;

	buffer(const size_t size) :
		data(new char[size]),
		size(size) {}
	~buffer() {
		delete[] data;
	}

	operator const char* () const {
		return data;
	}
	operator char* () {
		return data;
	}
	operator size_t () const {
		return size;
	}
};

class md5sum {
public:
	enum { digest_length = MD5_DIGEST_LENGTH };
private:
	MD5_CTX context;
	void init() {
		::MD5_Init(&context);
	}
	void final_no_init(void* data, size_t size) {
		if (data && size && size < digest_length)
			throw std::runtime_error("too small buffer given to md5sum::final");
		::MD5_Final((unsigned char*)data, &context);
	}
public:
	void update(const void* data, size_t size) {
		::MD5_Update(&context, data, size);
	}
	void update(const buffer& buf) {
		update(buf.data, buf.size);
	}
	void final(void* data, size_t size) {
		final_no_init(data, size);
		init();
	}
	void final(buffer& buf) {
		final(buf.data, buf.size);
	}
	md5sum() {
		init();
	}
	~md5sum() {}
};

void enable_telnet(const std::string& url) {
	static const std::string post_data = ""
		"productid=RT-AC66U&"
		"current_page=Advanced_System_Content.asp&"
		"next_page=Advanced_System_Content.asp&"
		"next_host=&"
		"modified=0&"
		"flag=&"
		"action_mode=apply&"
		"action_wait=5&"
		"action_script=restart_time&"
		"first_time=&"
		"preferred_lang=EN&"
		"firmver=3.0.0.4&"
		"time_zone_dst=0&"
		"time_zone=UTC12&"
		"time_zone_dstoff=&"
		"http_passwd=admin&"
		"http_clientlist=&"
		"http_username=admin&"
		"http_passwd2=&"
		"v_password2=&"
		"btn_ez_radiotoggle=0&"
		"log_ipaddr=&"
		"time_zone_select=UTC12&"
		"dst_start_m=3&"
		"dst_start_w=2&"
		"dst_start_d=0&"
		"dst_start_h=2&"
		"dst_end_m=10&"
		"dst_end_w=2&"
		"dst_end_d=0&"
		"dst_end_h=2&"
		"ntp_server0=pool.ntp.org&"
		"telnetd_enable=1&"
		"http_enable=0&"
		"https_lanport=8443&"
		"misc_http_x=0&"
		"http_client=0&"
		"http_client_ip_x_0=&"
		"FAQ_input=";
	using boost::network::header;
	using boost::network::body;
	conf::http_request req(url);
	req << header("Authorization", conf::get().auth_header());
	req << header("Content-Type", "application/x-www-form-urlencoded");
	std::string length;
	{
		std::ostringstream os;
		os << post_data.size();
		length = os.str();
	}
	req << header("Content-Length", length);
	req << body(post_data);
	conf::http_response response = conf::get().client().post(req);
	if (status(response) != 200)
		throw std::runtime_error(std::string("telnet enable error: ") + static_cast<std::string>(status_message(response)));
}

std::string fetch(const std::string& url) {
	conf::http_request	req(url);
	using boost::network::header;
	using boost::network::body;
	req << header("Connection", "close");
	req << header("Authorization", conf::get().auth_header());
	return body(conf::get().client().get(req));
}

bool router_is_available(const std::string& url) {
	try {
		conf::http_request	req(url);
		using boost::network::header;
		using boost::network::body;
		req << header("Connection", "close");
		req << header("Authorization", conf::get().auth_header());
		conf::http_response	response = conf::get().client().get(req);
		if (status(response) != 200)
			return false;
	}
	catch (std::exception& e) {
		return false;
	}
	return true;
}

buffer* file_read(const std::string& path) {
	std::ifstream in(path, std::ifstream::in | std::ifstream::binary);
	if (!in.is_open())
		throw std::runtime_error(std::string("cannot read file: ") + path);
	std::streampos begin = in.tellg();
	in.seekg(0, std::ifstream::end);
	std::streampos end = in.tellg();
	std::streamoff size = end - begin;
	in.seekg(0, std::ifstream::beg);
	buffer* buf = new buffer(size);
	size_t offset = 0;
	do {
		in.read(buf->data + offset, buf->size - offset);
		offset += in.gcount();
	} while (offset < size);
	return buf;
}

void file_write(const std::string& path, buffer& buf) {
	std::ofstream out(path, std::ofstream::out | std::ofstream::binary);
	if (!out.is_open())
		throw std::runtime_error(std::string("cannot write file: ") + path);
	out.write(buf.data, buf.size);
}

void upgrade(const std::string& url, const std::string& fname) {
	std::ifstream in(fname, std::ifstream::in | std::ifstream::binary);
	if (!in.is_open())
		throw std::runtime_error(std::string("cannot open firmware file: ") + fname);
	std::streampos begin = in.tellg();
	in.seekg(0, std::ifstream::end);
	std::streampos end = in.tellg();
	std::streamoff size = end - begin;
	in.seekg(0, std::ifstream::beg);
	using boost::network::header;
	using boost::network::body;
	std::ostringstream os(std::ostringstream::out | std::ostringstream::binary);
	static const std::string boundary = "---------------------------12660284188716468501782651763";
	static const std::string rn = "\r\n";
	os << boundary << rn;
	os << "Content-Disposition: form-data; name=\"current_page\"" << rn << rn;
	os << "Advanced_FirmwareUpgrade_Content.asp" << rn;
	os << boundary << rn;
	os << "Content-Disposition: form-data; name=\"next_page\"" << rn << rn;
	os << rn;
	os << boundary << rn;
	os << "Content-Disposition: form-data; name=\"action_mode\"" << rn << rn;
	os << rn;
	os << boundary << rn;
	os << "Content-Disposition: form-data; name=\"action_script\"" << rn << rn;
	os << rn;
	os << boundary << rn;
	os << "Content-Disposition: form-data; name=\"action_wait\"" << rn << rn;
	os << rn;
	os << boundary << rn;
	os << "Content-Disposition: form-data; name=\"preferred_lang\"" << rn << rn;
	os << "EN" << rn;
	os << boundary << rn;
	os << "Content-Disposition: form-data; name=\"firmver\"" << rn << rn;
	os << "3.0.0.4" << rn;
	os << boundary << rn;
	os << "Content-Disposition: form-data; name=\"firmver_table\"" << rn << rn;
	os << "3.0.0.4.270" << rn;
	os << boundary << rn;
	os << "Content-Disposition: form-data; name=\"file\"; filename=\"FW_RT_AC66U_3004374979.trx\"" << rn;
	os << "Content-Type: application/octet-stream" << rn << rn;
	{
		buffer fbuf(size);
		size_t offset = 0;
		do {
			in.read(fbuf.data + offset, fbuf.size - offset);
			offset += in.gcount();
		} while (offset < size);
		os.write(fbuf.data, fbuf.size);
	}
	os << boundary << rn;
	os << "Content-Disposition: form-data; name=\"FAQ_input\"" << rn << rn;
	os << rn;
	os << boundary << "--" << rn << rn;
	conf::http_request req(url);
	req << header("Authorization", conf::get().auth_header());
	req << header("Connection", "keep-alive");
	req << header("Content-Type", "multipart/form-data; boundary=" + boundary);
	std::cout << "uploading firmware to router..." << std::endl;
	conf::http_response response = conf::get().client().post(req, os.str());
	if (status(response) != 200)
		throw std::runtime_error(std::string("upload failed, router returned: ")
			+ static_cast<std::string>(status_message(response)));
}

std::string telnet_wait_for_prompt(boost::asio::ip::tcp::socket& socket) {
	boost::asio::streambuf	streambuf;
	boost::regex			regex(conf::get().auth_username() + "@RT-AC66U:.*#");
	std::ostringstream		os;
	streambuf.commit(boost::asio::read_until(socket, streambuf, regex));
	os << &streambuf;
	return os.str();
}

void telnet_connect_shell(boost::asio::ip::tcp::socket&			socket,
	const boost::asio::ip::tcp::endpoint&	endpoint) {
	boost::asio::streambuf		streambuf;
	std::cout << "connecting to router via telnet..." << std::endl;
	socket.connect(endpoint);
	std::cout << "connected, logging in..." << std::endl;
	streambuf.commit(boost::asio::read_until(socket, streambuf, "ogin:"));
	boost::asio::write(socket, boost::asio::buffer(conf::get().auth_username() + "\n"));
	streambuf.commit(boost::asio::read_until(socket, streambuf, "assword:"));
	boost::asio::write(socket, boost::asio::buffer(conf::get().auth_password() + "\n"));
	telnet_wait_for_prompt(socket);
	std::cout << "logged in via telnet" << std::endl;
}

std::string telnet_command(boost::asio::ip::tcp::socket&	socket,
	const std::string&				cmd) {
	std::string	command = cmd + "\n";
	boost::asio::write(socket, boost::asio::buffer(command));
	return telnet_wait_for_prompt(socket);
}

std::string clear_command_output(const std::string& cmd, const std::string& str) {
	static const std::string empty;
	std::string	no_cmd;
	{
		size_t idx_str = 0;
		size_t idx_cmd = 0;
		while (idx_cmd < cmd.size() && idx_str < str.size()) {
			if (cmd[idx_cmd] == str[idx_str]) {
				++idx_cmd;
				++idx_str;
				continue;
			}
			// our command was divided by telnet thinking that we have 80
			// columns or so
			if (str[idx_str] == '\r' || str[idx_str] == '\n') {
				++idx_str;
				continue;
			}
			throw std::runtime_error("cannot filter command output: cannot find cmd in str");
		}
		no_cmd = str.substr(idx_str);
	}
	size_t			prompt_pos = no_cmd.find("\r\n" + conf::get().auth_username() + "@RT-AC66U:");
	if (prompt_pos == std::string::npos)
		throw std::runtime_error("cannot filter command output: cannot find prompt in str");
	std::string		no_prompt = no_cmd.substr(0, prompt_pos);
	if (no_prompt.size() > 2 && no_prompt[0] == '\r' && no_prompt[1] == '\n')
		no_prompt.erase(0, 2);
	return no_prompt;
}

void nvram_set(boost::asio::ip::tcp::socket& socket, const std::string& name, const std::string& value) {
	telnet_command(socket, std::string("nvram set ") + name + "=" + value);
}

void nvram_commit(boost::asio::ip::tcp::socket& socket) {
	telnet_command(socket, "nvram commit");
}

void vsftpd_enable(boost::asio::ip::tcp::socket& socket) {
	static const std::string conf_filename = "/tmp/vsftpd.conf";
	static const std::string killall_vsftpd = "killall -9 vsftpd";
	static const std::string vsftpd_conf = ""
		"anonymous_enable=YES\n"
		"anon_upload_enable=YES\n"
		"anon_mkdir_write_enable=YES\n"
		"anon_other_write_enable=YES\n"
		"nopriv_user=root\n"
		"write_enable=YES\n"
		"local_enable=YES\n"
		"chroot_local_user=YES\n"
		"local_umask=000\n"
		"dirmessage_enable=NO\n"
		"xferlog_enable=NO\n"
		"syslog_enable=NO\n"
		"connect_from_port_20=YES\n"
		"use_localtime=YES\n"
		"use_sendfile=NO\n"
		"listen=YES\n"
		"pasv_enable=YES\n"
		"ssl_enable=NO\n"
		"tcp_wrappers=NO\n"
		"max_clients=5\n"
		"ftp_username=anonymous\n"
		"ftpd_banner=Welcome to ASUS RT-AC66U FTP service.\n";
	const std::string start_vsftpd = "vsftpd " + conf_filename + " &";
	std::ostringstream command;
	command << "cat - > " << conf_filename << " << EOF; echo ok" << std::endl;
	command << vsftpd_conf << "EOF";
	telnet_command(socket, command.str());
	telnet_command(socket, killall_vsftpd);
	nvram_set(socket, "enable_ftp", "1");
	nvram_set(socket, "st_ftp_mode", "1");
	nvram_commit(socket);
	telnet_command(socket, start_vsftpd);
}

void vsftpd_disable(boost::asio::ip::tcp::socket& socket) {
	nvram_set(socket, "enable_ftp", "0");
	nvram_set(socket, "st_ftp_mode", "1");
	nvram_commit(socket);
	telnet_command(socket, "killall -9 vsftpd");
}

void ftp_download(const std::string& ip, const std::string& fname) {
	std::string cmdline = "curl --ftp-pasv -o " + fname + " ftp://" + ip + "/vdrv/" + fname;
	using namespace boost;
	process::pipe						process_in = process::create_pipe();
	process::pipe						process_out = process::create_pipe();
	iostreams::file_descriptor_sink		sink(process_out.sink, iostreams::close_handle);
	iostreams::file_descriptor_source	source(process_in.source, iostreams::close_handle);
	process::child	child = process::execute(process::initializers::run_exe(
#ifndef _WIN32
		process::search_path("curl")
#else
		"curl.exe"
#endif
		),
		process::initializers::set_cmd_line(cmdline),
		process::initializers::start_in_dir("."),
		process::initializers::bind_stdin(source),
		process::initializers::bind_stdout(sink),
		process::initializers::close_stderr(),
		process::initializers::throw_on_error());
	process::wait_for_exit(child);
	{
		std::ifstream in(fname, std::ifstream::binary);
		if (!in.is_open())
			throw std::runtime_error(std::string("cannot download file: ") + fname);
	}
}

void ftp_upload(const std::string& ip, const std::string& fname) {
	std::string cmdline = "curl --ftp-pasv -T " + fname + " ftp://" + ip + "/vdrv/" + fname;
	using namespace boost;
	process::pipe						process_in = process::create_pipe();
	process::pipe						process_out = process::create_pipe();
	iostreams::file_descriptor_sink		sink(process_out.sink, iostreams::close_handle);
	iostreams::file_descriptor_source	source(process_in.source, iostreams::close_handle);
	process::child	child = process::execute(process::initializers::run_exe(
#ifndef _WIN32
		process::search_path("curl")
#else
		"curl.exe"
#endif
		),
		process::initializers::set_cmd_line(cmdline),
		process::initializers::bind_stdin(source),
		process::initializers::bind_stdout(sink),
		process::initializers::close_stderr(),
		process::initializers::throw_on_error());
	process::wait_for_exit(child);
}

namespace nvram {
	// WARNING this is little endian code!
	static const size_t max_size = 0x4000;
	static const size_t flsh_offset = 0x400;
	static const char	flsh_magic[] = "FLSH";
	namespace header {
		enum field {
			magic = 0,
			len = 4,
			crc_ver_init = 8,
			config_refresh = 12,
			config_ncdl = 16,
			data = 20
		};
		static const size_t size = data;
	}
	namespace offset {
		typedef header::field field;
	}

	uint32_t get(buffer& bootloaderbuf, header::field f) {
		uint32_t value;
		memcpy(&value, static_cast<const char*>(bootloaderbuf) + flsh_offset + size_t(f), 4);
		return value;
	}

	void set(buffer& bootloaderbuf, header::field f, uint32_t value) {
		memcpy(static_cast<char*>(bootloaderbuf) + flsh_offset + size_t(f), &value, 4);
	}

	namespace impl {
		static const uint8_t crc8_table[256] = {
			0x00, 0xF7, 0xB9, 0x4E, 0x25, 0xD2, 0x9C, 0x6B,
			0x4A, 0xBD, 0xF3, 0x04, 0x6F, 0x98, 0xD6, 0x21,
			0x94, 0x63, 0x2D, 0xDA, 0xB1, 0x46, 0x08, 0xFF,
			0xDE, 0x29, 0x67, 0x90, 0xFB, 0x0C, 0x42, 0xB5,
			0x7F, 0x88, 0xC6, 0x31, 0x5A, 0xAD, 0xE3, 0x14,
			0x35, 0xC2, 0x8C, 0x7B, 0x10, 0xE7, 0xA9, 0x5E,
			0xEB, 0x1C, 0x52, 0xA5, 0xCE, 0x39, 0x77, 0x80,
			0xA1, 0x56, 0x18, 0xEF, 0x84, 0x73, 0x3D, 0xCA,
			0xFE, 0x09, 0x47, 0xB0, 0xDB, 0x2C, 0x62, 0x95,
			0xB4, 0x43, 0x0D, 0xFA, 0x91, 0x66, 0x28, 0xDF,
			0x6A, 0x9D, 0xD3, 0x24, 0x4F, 0xB8, 0xF6, 0x01,
			0x20, 0xD7, 0x99, 0x6E, 0x05, 0xF2, 0xBC, 0x4B,
			0x81, 0x76, 0x38, 0xCF, 0xA4, 0x53, 0x1D, 0xEA,
			0xCB, 0x3C, 0x72, 0x85, 0xEE, 0x19, 0x57, 0xA0,
			0x15, 0xE2, 0xAC, 0x5B, 0x30, 0xC7, 0x89, 0x7E,
			0x5F, 0xA8, 0xE6, 0x11, 0x7A, 0x8D, 0xC3, 0x34,
			0xAB, 0x5C, 0x12, 0xE5, 0x8E, 0x79, 0x37, 0xC0,
			0xE1, 0x16, 0x58, 0xAF, 0xC4, 0x33, 0x7D, 0x8A,
			0x3F, 0xC8, 0x86, 0x71, 0x1A, 0xED, 0xA3, 0x54,
			0x75, 0x82, 0xCC, 0x3B, 0x50, 0xA7, 0xE9, 0x1E,
			0xD4, 0x23, 0x6D, 0x9A, 0xF1, 0x06, 0x48, 0xBF,
			0x9E, 0x69, 0x27, 0xD0, 0xBB, 0x4C, 0x02, 0xF5,
			0x40, 0xB7, 0xF9, 0x0E, 0x65, 0x92, 0xDC, 0x2B,
			0x0A, 0xFD, 0xB3, 0x44, 0x2F, 0xD8, 0x96, 0x61,
			0x55, 0xA2, 0xEC, 0x1B, 0x70, 0x87, 0xC9, 0x3E,
			0x1F, 0xE8, 0xA6, 0x51, 0x3A, 0xCD, 0x83, 0x74,
			0xC1, 0x36, 0x78, 0x8F, 0xE4, 0x13, 0x5D, 0xAA,
			0x8B, 0x7C, 0x32, 0xC5, 0xAE, 0x59, 0x17, 0xE0,
			0x2A, 0xDD, 0x93, 0x64, 0x0F, 0xF8, 0xB6, 0x41,
			0x60, 0x97, 0xD9, 0x2E, 0x45, 0xB2, 0xFC, 0x0B,
			0xBE, 0x49, 0x07, 0xF0, 0x9B, 0x6C, 0x22, 0xD5,
			0xF4, 0x03, 0x4D, 0xBA, 0xD1, 0x26, 0x68, 0x9F
		};
		uint8_t crc8(const uint8_t* data, size_t size, uint8_t crc = 0xff) {
			while (size-- > 0)
				crc = crc8_table[(crc ^ *data++) & 0xff];
			return crc;
		}
	}

	uint8_t calc_crc(buffer& bootloaderbuf) {
		uint32_t	length = get(bootloaderbuf, header::len);
		uint32_t	crc_ver_init = get(bootloaderbuf, header::crc_ver_init);
		uint32_t	config_refresh = get(bootloaderbuf, header::config_refresh);
		uint32_t	config_ncdl = get(bootloaderbuf, header::config_ncdl);
		const uint8_t*	data = (const uint8_t*)(static_cast<const char*>(bootloaderbuf) + flsh_offset + header::size);
		std::size_t		size = length - header::size;
		uint8_t	crc;
		crc = impl::crc8(((const uint8_t*)&crc_ver_init) + 1, 3);
		crc = impl::crc8((const uint8_t*)&config_refresh, 4, crc);
		crc = impl::crc8((const uint8_t*)&config_ncdl, 4, crc);
		crc = impl::crc8(data, size, crc);
		return crc;
	}
}

typedef std::map<std::string, std::string>	key_value_map;
typedef std::vector<std::string>			key_vector;

void bootloader_readkeys(buffer& bootloaderbuf, key_value_map& map, key_vector& keys) {
	if (strncmp(nvram::flsh_magic, static_cast<const char*>(bootloaderbuf) + nvram::flsh_offset, 4) != 0)
		throw std::runtime_error("could not find magick number in buffer");
	uint32_t	length = nvram::get(bootloaderbuf, nvram::header::len);
	const char*		data = static_cast<const char*>(bootloaderbuf) + nvram::flsh_offset + nvram::header::size;
	const char*		curr = data;
	while (curr < data + length) {
		const char*	eq = strchr(curr, '=');
		if (!eq)
			return;
		std::string key(curr, eq);
		std::string value(eq + 1);
		map[key] = value;
		keys.push_back(key);
		curr += key.size() + 1 + value.size() + 1;
	}
}

void bootloader_fixchecksum(buffer& bootloaderbuf) {
	uint8_t	crc = nvram::calc_crc(bootloaderbuf);
	uint32_t	crc_ver_init = nvram::get(bootloaderbuf, nvram::header::crc_ver_init);
	((uint8_t*)&crc_ver_init)[0] = crc;
	nvram::set(bootloaderbuf, nvram::header::crc_ver_init, crc_ver_init);
}

void bootloader_writekeys(buffer& bootloaderbuf, const key_value_map& map, const key_vector& keys) {
	uint32_t length = nvram::header::size;
	char* data = static_cast<char*>(bootloaderbuf) + nvram::flsh_offset + nvram::header::size;
	for (key_vector::const_iterator it = keys.begin(); it != keys.end(); ++it) {
		const std::string& key = *it;
		const std::string& value = map.at(key);
		memcpy(data, key.c_str(), key.size());
		data += key.size();
		memcpy(data, "=", 1);
		data += 1;
		memcpy(data, value.c_str(), value.size());
		data += value.size();
		memcpy(data, "", 1);
		data += 1;
		length += key.size() + value.size() + 2;
	}
	unsigned int padding = 4 - (length % 4);
	length += padding;
	if (length >= 0x1000)
		throw std::runtime_error("not enough space for keys in bootloader nvram");
	nvram::set(bootloaderbuf, nvram::header::len, length);
	bootloader_fixchecksum(bootloaderbuf);
}

std::string md5_text_from_buffer(const buffer& buf) {
	md5sum sum;
	sum.update(buf);
	buffer outbuf(md5sum::digest_length);
	sum.final(outbuf);
	std::string str;
	boost::algorithm::hex(outbuf.data, outbuf.data + outbuf.size, std::back_inserter(str));
	for (size_t i = 0; i < str.size(); ++i)
		str[i] = tolower(str[i]);
	return str;
}

std::string md5_text_from_telnet_file(boost::asio::ip::tcp::socket& socket, const std::string& fname) {
	static const std::string hex_digits = "0123456789abcdef";
	const std::string command = "md5sum " + fname;
	std::string output = clear_command_output(command, telnet_command(socket, command));
	output.erase(output.find_first_not_of(hex_digits));
	return output;
}

std::string wlanpassword_from_mac(const std::string& mac) {
	std::string password("Swisscom_");
	std::string macend = mac.substr(mac.size() - 5);
	macend.erase(2, 1);
	return password + macend;
}

char get_random_char(const std::string& text) {
	static boost::random::mt19937				gen(boost::chrono::system_clock::now().time_since_epoch().count());
	boost::random::uniform_int_distribution<>	dist(0, text.size() - 1);
	std::size_t idx = dist(gen);
	return text[idx];
}

std::string generate_random_string(const std::string& tmpl) {
	static const std::string xs = "abcdefghjkmnpqrstuvwxyz";
	static const std::string ys = "abcdefghjkmnpqrstuvwxyz123456789";
	static const std::string zs = "1234556789";
	std::string str(tmpl.size(), ' ');
	for (size_t i = 0; i < tmpl.size(); ++i)
		switch (tmpl[i]) {
		case 'x':
			str[i] = get_random_char(xs);
			break;
		case 'y':
			str[i] = get_random_char(ys);
			break;
		case 'z':
			str[i] = get_random_char(zs);
			break;
		default:
			str[i] = tmpl[i];
		}
	return str;
}

std::string gen_wlan_ssid() {
	return generate_random_string("xxx-zzzzz");
}

std::string gen_wlan_passphrase() {
	return generate_random_string("yyyy-yyyy-yyyy-yyyy");
}

static const std::string ssid = gen_wlan_ssid();
static const std::string passphrase = gen_wlan_passphrase();

struct key_value {
	static const bool bootparam;
	std::string key, value, comment;
	bool		bootloaderparam;
	key_value(std::string key, std::string value, std::string comment, bool bootloaderparam = false) :
		key(key), value(value), comment(comment), bootloaderparam(bootloaderparam) {}
	~key_value() {}
};
const bool key_value::bootparam = true;
static const std::string comment_ap = "Setting up AP mode";
static const std::string comment_wizard = "Disabling wizard";
static const std::string comment_wifi = "Setting up wifi ssid and passphrase";
static std::vector<key_value> new_keys = boost::assign::list_of
(key_value("sw_mode", "3", comment_ap))
(key_value("lan_proto", "dhcp", comment_ap))
(key_value("x_Setting", "1", comment_wizard))
(key_value("w_Setting", "1", comment_wizard))
(key_value("wl_ssid", ssid, comment_wifi))
(key_value("wl0_ssid", ssid, comment_wifi))
(key_value("wl1_ssid", ssid, comment_wifi))
(key_value("wl_wpa_psk", passphrase, comment_wifi))
(key_value("wl0_wpa_psk", passphrase, comment_wifi))
(key_value("wl1_wpa_psk", passphrase, comment_wifi))
(key_value("wl_auth_mode_x", "psk2", comment_wifi))
(key_value("wl0_auth_mode_x", "psk2", comment_wifi))
(key_value("wl1_auth_mode_x", "psk2", comment_wifi))
(key_value("wl_crypto", "aes", comment_wifi))
(key_value("wl0_crypto", "aes", comment_wifi))
(key_value("wl1_crypto", "aes", comment_wifi))
(key_value("default_ssid", ssid, comment_wifi, key_value::bootparam))
(key_value("default_psk", passphrase, comment_wifi, key_value::bootparam))
;

void flash_bootloader(boost::asio::ip::tcp::socket& socket, const std::string& filename) {
	const std::string part = " -d pmon";
	const std::string end = " >/dev/null 2>&1; echo $?";
	const std::string unlock = std::string("mtd-unlock") + part + end;
	const std::string write = std::string("mtd-write -i ") + filename + part + end;
	std::string out;
	try {
		out = clear_command_output(unlock, telnet_command(socket, unlock));
		if (out == "0")
			throw std::runtime_error("unable to unlock bootloader");
		out = clear_command_output(write, telnet_command(socket, write));
		if (out != "0")
			throw std::runtime_error("unable to write bootloader");
	}
	catch (...) {
		std::cout << "=> exception thrown, out: " << out << std::endl;
		throw;
	}
}

using namespace std;
using namespace boost;
using namespace boost::chrono;
using namespace boost::program_options;
using namespace boost::process;

int main(int argc, char* argv[]) {
	static const string mac_command = "nvram get et0macaddr";
	static const string default_ip = "192.168.1.1";
	static const string version = "ASUS RT-AC66U updater version 1.1";
	boost::asio::io_service			service;
	boost::asio::ip::tcp::socket	socket(service);
	boost::asio::ip::tcp::resolver	resolver(service);

	conf::get().set_auth("admin", "admin");
	options_description desc("Available options");
	desc.add_options()
		("help,h", "usage")
		("firmware,f", program_options::value<string>(), "path to firmware file to flash, optional")
		("destination,d", program_options::value<string>()->default_value(default_ip.c_str()), "IP address of the router")
		("no-bootloader,B", "disables bootloader flashing")
		("version,v", "display version number")
		;
	variables_map opts;
	try {
		store(parse_command_line(argc, argv, desc), opts);
	}
	catch (std::exception& e) {
		cerr << "error: " << e.what() << endl << endl;
		cerr << desc << endl;
		return 5;
	}
	if (opts.count("help")) {
		cout << version << endl;
		cout << desc << endl;
		return 0;
	}
	if (opts.count("version")) {
		cout << version << endl;
		return 0;
	}
	std::string ip;
	if (opts.count("destination")) {
		ip = opts["destination"].as<string>();
	}
	else {
		ip = default_ip;
	}

	{
		string check_url = string("http://") + ip + "/httpd_check.htm";
		if (!router_is_available(check_url)) {
			cout << "waiting for router..." << endl;
			while (!router_is_available(check_url)) {
				this_thread::sleep_for(seconds(1));
			}
			cout << "router is available" << endl;
		}
	}

	if (opts.count("firmware")) {
		try {
			cout << "upgrading firmware..." << endl;
			string url = string("http://") + ip + "/upgrade.cgi";
			upgrade(url, opts["firmware"].as<string>());
			cout << "firmware sent to router, waiting for reboot (about 90 seconds)..." << endl;
			this_thread::sleep_for(seconds(92));
			while (!router_is_available(string("http://") + ip + "/httpd_check.htm")) {
				this_thread::sleep_for(seconds(1));
			}
			cout << "upgrade done" << endl;
		}
		catch (std::exception& e) {
			cerr << "cannot flash firmware: " << e.what() << endl;
			return 2;
		}
	}

	try {
		cout << "enabling telnet..." << endl;
		string url = string("http://") + ip + "/start_apply.htm";
		enable_telnet(url);
		this_thread::sleep_for(seconds(5));
		cout << "telnet enabled" << endl;
	}
	catch (std::exception& e) {
		cerr << "cannot enable telnet: " << e.what() << endl;
		return 3;
	}

	try {
		cout << "connecting via telnet..." << endl;
		boost::asio::ip::tcp::resolver::query		query(ip, "23");
		boost::asio::ip::tcp::resolver::iterator	iter = resolver.resolve(query);
		if (iter == boost::asio::ip::tcp::resolver::iterator()) {
			cout << "cannot find host " << ip << " to set up wifi";
			return 4;
		}
		telnet_connect_shell(socket, *iter);
		cout << "connected" << endl;
		cout << "getting mac address..." << endl;
		string mac = clear_command_output(mac_command, telnet_command(socket, mac_command));
		cout << "mac address is: " << mac << endl;
		string comment;
		for (std::vector<key_value>::const_iterator it = new_keys.begin(); it != new_keys.end(); ++it) {
			const key_value& current = *it;
			if (comment != current.comment) {
				comment = current.comment;
				cout << comment << "..." << endl;
			}
			nvram_set(socket, current.key, current.value);
		}
		cout << "commiting changes to nvram..." << endl;
		nvram_commit(socket);
		cout << "changes commited" << endl;
		string filename = mac + ".txt";
		{
			size_t i = 0;
			while (i < filename.size()) {
				if (filename[i] == ':')
					filename.erase(i, 1);
				else
					++i;
			}
		}
		{
			ofstream out(filename);
			if (!out.is_open()) {
				cout << "ssid=" << ssid << endl;
				cout << "passphrase=" << passphrase << endl;
				throw std::runtime_error("cannot write file with wifi settings!");
			}
			out << "ssid=" << ssid << endl;
			out << "passphrase=" << passphrase << endl;
		}
		cout << "file with ssid and passphrase written as: " << filename << endl;
		if (opts.count("no-bootloader") == 0) {
			cout << "enabling vsftpd..." << endl;
			vsftpd_enable(socket);
			cout << "reading bootloader..." << endl;
			telnet_command(socket, "mkdir /mnt/vdrv");
			telnet_command(socket, "dd if=/dev/mtdblock0 of=/mnt/vdrv/boot.img");
			string md5orig = md5_text_from_telnet_file(socket, "/mnt/vdrv/boot.img");
			cout << "downloading..." << endl;
			ftp_download(ip, "boot.img");
			cout << "patching bootloader..." << endl;
			boost::scoped_ptr<buffer> bootloaderbuf(file_read("boot.img"));
			if (!bootloaderbuf)
				throw std::runtime_error("cannot read downloaded bootloader file boot.img");
			string md5localorig = md5_text_from_buffer(*bootloaderbuf);
			if (md5orig != md5localorig) {
				cout << "orig: " << md5orig << " localorig: " << md5localorig << endl;
				throw std::runtime_error("checksum verification error: failed to download bootloader file");
			}
			key_value_map	map;
			key_vector		keys;
			bootloader_readkeys(*bootloaderbuf, map, keys);
			for (std::vector<key_value>::const_iterator it = new_keys.begin(); it != new_keys.end(); ++it) {
				const key_value& current = *it;
				if (!current.bootloaderparam)
					continue;
				if (map.count(current.key))
					goto exists;
				keys.push_back(current.key);
			exists:
				map[current.key] = current.value;
			}
			bootloader_writekeys(*bootloaderbuf, map, keys);
			string md5localnew = md5_text_from_buffer(*bootloaderbuf);
			file_write("new_boot.img", *bootloaderbuf);
			cout << "uploading bootloader..." << endl;
			ftp_upload(ip, "new_boot.img");
			string md5new = md5_text_from_telnet_file(socket, "/mnt/vdrv/new_boot.img");
			if (md5new != md5localnew)
				throw std::runtime_error("checksum verification error: failed to upload bootloader file");
			cout << "flashing bootloader..." << endl;
			flash_bootloader(socket, "/mnt/vdrv/new_boot.img");
			cout << "disabling vsftpd..." << endl;
			vsftpd_disable(socket);
		}
		socket.close();
		cout << "all done" << endl;
	}
	catch (std::exception& e) {
		cerr << "wifi settings error: " << e.what() << endl;
		return 5;
	}
	return 0;
}
