/****************************************************************************
**
** Copyright (C) 2011 SoftAtHome. All rights reserved.
**
** SoftAtHome reserves all rights not expressly granted herein.
**
** - DISCLAIMER OF WARRANTY -
**
** THIS FILE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
** EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE.
**
** THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOURCE
** CODE IS WITH YOU. SHOULD THE SOURCE CODE PROVE DEFECTIVE, YOU
** ASSUME THE COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
**
** - LIMITATION OF LIABILITY -
**
** IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN
** WRITING WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MODIFIES
** AND/OR DISTRIBUTES THE SOURCE CODE, BE LIABLE TO YOU FOR DAMAGES,
** INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES
** ARISING OUT OF THE USE OR INABILITY TO USE THE SOURCE CODE
** (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING RENDERED
** INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE
** OF THE SOURCE CODE TO OPERATE WITH ANY OTHER PROGRAM), EVEN IF SUCH
** HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
** DAMAGES.
**
****************************************************************************/

#include <signal.h>

#include <debug/sahtrace.h>

#include <pcb/core.h>
#include <pcb/utils.h>
#include <pcb/pcb_client.h>

#include "common.h"

static client_config_t *cfg = NULL;
static request_t *request = NULL;

bool pcb_client_handle_reply_item(request_t __attribute__((unused)) *req,
                       reply_item_t *item,
                       pcb_t __attribute__((unused)) *pcb,
                       peer_info_t __attribute__((unused)) *from,
                       void __attribute__((unused)) *userdata) {
    SAH_TRACE_IN();

    switch(reply_item_type(item)) {
        case reply_type_object:
            print_object(reply_item_object(item));
        break;
        case reply_type_notification:
            print_notification(reply_item_notification(item));
        break;
        case reply_type_error:
            SAH_TRACE_ERROR("Request failed, exiting - error - %d - %s - %s",reply_item_error(item),
                                                                             reply_item_errorDescription(item),
                                                                             reply_item_errorInfo(item));
            connection_setTerminated(true);
            SAH_TRACE_OUT();
            return false;
        break;
        default:
        break;
    }

    SAH_TRACE_OUT();
    return true;
}

static request_t *pcb_client_create_send_request(peer_info_t *destination, const char *object_path) {
    // 1. create the request.
    request_t *request = request_create_getObject(object_path,                      // object path
                                                  -1,                                // request depth
                                                  request_getObject_all |           // request flags
                                                  request_notify_all |
                                                  request_no_object_caching);
    if (!request) {
        SAH_TRACE_ERROR("Failed to create request (0x%8.8X - %s), exiting", pcb_error, error_string(pcb_error));
        goto exit_error;
    }

    // 2. Set the reply handler
    request_setReplyItemHandler(request, pcb_client_handle_reply_item);

    // 3. Send the request
    if (!pcb_sendRequest(cfg->pcb, destination, request)) {
        SAH_TRACE_ERROR("Failed to send request (0x%8.8X - %s), exiting", pcb_error, error_string(pcb_error));
        goto exit_error_free_request;
    }

    return request;

exit_error_free_request:
    request_destroy(request);

exit_error:
    return NULL;
}

bool pcb_client_initialize(pcb_t *pcb, int argc, char *argv[]) {
    SAH_TRACE_IN();
    cfg = pcb_clientConfig(pcb);

    // 1. verify arguments:
    // usage: OBJECT_PATH
    if (argc < 1) {
        SAH_TRACE_ERROR("Missing arguments, exiting");
        fprintf(stderr, "Usage OBJECT_PATH\n");
        goto exit_error;
    }

    // 2. create and send a request.
    request = pcb_client_create_send_request(cfg->system_bus, argv[0]);
    if (!request) {
        SAH_TRACE_ERROR("Failed to create and send request");
        goto exit_error;
    }

    SAH_TRACE_OUT();
    return true;

exit_error:
    SAH_TRACE_OUT();
    return false;
}

void pcb_client_cleanup() {
    SAH_TRACE_IN();

    request_destroy(request);
    SAH_TRACE_OUT();
}
