/****************************************************************************
**
** Copyright (C) 2011 SoftAtHome. All rights reserved.
**
** SoftAtHome reserves all rights not expressly granted herein.
**
** - DISCLAIMER OF WARRANTY -
**
** THIS FILE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
** EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE.
**
** THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOURCE
** CODE IS WITH YOU. SHOULD THE SOURCE CODE PROVE DEFECTIVE, YOU
** ASSUME THE COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
**
** - LIMITATION OF LIABILITY -
**
** IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN
** WRITING WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MODIFIES
** AND/OR DISTRIBUTES THE SOURCE CODE, BE LIABLE TO YOU FOR DAMAGES,
** INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES
** ARISING OUT OF THE USE OR INABILITY TO USE THE SOURCE CODE
** (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING RENDERED
** INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE
** OF THE SOURCE CODE TO OPERATE WITH ANY OTHER PROGRAM), EVEN IF SUCH
** HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
** DAMAGES.
**
****************************************************************************/

#ifndef __URI_H__
#define __URI_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include <pcb/common/error.h>

/**
@ingroup pcb_utils
@file
@brief
Header file with uri type definitions and uri functions
*/

/**
@ingroup pcb_utils
@defgroup pcb_utils_uri URI Parsing
@{

@details
This utility provides a simple uti parser. It can parse URIs of the following format.
@image html url.png

Also a path can be added to the uri and has then the following from.
@image html url-path.png

Parsing an uri string can be done by calling @ref uri_parse. This function will return
a pointer to a @ref uri_t structure. You need to free that structure again by calling
@ref uri_destroy.

The structure definition is public available and is directly accesible. But this way of working
is not encouraged, use the accessor functions instead.

*/

/**
@brief
URI structure definition

@details
This structure defines a parsed uri.
*/
typedef struct _uri {
    char *scheme;    /**< The uri scheme */
    char *host;      /**< The uri host */
    char *port;      /**< The uri port */
    char *user;      /**< The uri user */
    char *password;  /**< The uri password */
    char *path;      /**< The uri path */
    int   parametersLength; /**< Length of the string containing the parameters */
    char *parameters; /**< The parameters */
} uri_t;

/**
@}
*/

uri_t *uri_parse(const char *URI);
void uri_destroy(uri_t *URI);

/**
@ingroup pcb_utils_uri
@brief Get the uri scheme
@details Get the uri scheme
@param URI pointer to the uri_t structure
@return The string representing the scheme. Do not free the returned string, use @ref uri_destroy.
*/
static inline const char *uri_getScheme(const uri_t *URI) {
    return URI?URI->scheme:NULL;
}

/**
@ingroup pcb_utils_uri
@brief Get the uri host
@details Get the uri host
@param URI pointer to the uri_t structure
@return The string representing the host. Do not free the returned string, use @ref uri_destroy.
*/
static inline const char *uri_getHost(const uri_t *URI) {
    return URI?URI->host:NULL;
}

/**
@ingroup pcb_utils_uri
@brief Get the uri port
@details Get the uri port
@param URI pointer to the uri_t structure
@return The string representing the port. Do not free the returned string, use @ref uri_destroy.
*/
static inline const char *uri_getPort(const uri_t *URI) {
    return URI?URI->port:NULL;
}

/**
@ingroup pcb_utils_uri
@brief Get the uri user
@details Get the uri user
@param URI pointer to the uri_t structure
@return The string representing the user. Do not free the returned string, use @ref uri_destroy.
*/
static inline const char *uri_getUser(const uri_t *URI) {
    return URI?URI->user:NULL;
}

/**
@ingroup pcb_utils_uri
@brief Get the uri password
@details Get the uri password
@param URI pointer to the uri_t structure
@return The string representing the password. Do not free the returned string, use @ref uri_destroy.
*/
static inline const char *uri_getPassword(const uri_t *URI) {
    return URI?URI->password:NULL;
}

/**
@ingroup pcb_utils_uri
@brief Get the uri path
@details Get the uri path
@param URI pointer to the uri_t structure
@return The string representing the path. Do not free the returned string, use @ref uri_destroy.
*/
static inline const char *uri_getPath(const uri_t *URI) {
    return URI?URI->path:NULL;
}

const char *uri_getParameter(const uri_t *URI, const char *name);
char *uri_string(const uri_t *URI);

#ifdef __cplusplus
}
#endif
#endif
