/****************************************************************************
**
** Copyright (C) 2011 SoftAtHome. All rights reserved.
**
** SoftAtHome reserves all rights not expressly granted herein.
**
** - DISCLAIMER OF WARRANTY -
**
** THIS FILE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
** EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE.
**
** THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOURCE
** CODE IS WITH YOU. SHOULD THE SOURCE CODE PROVE DEFECTIVE, YOU
** ASSUME THE COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
**
** - LIMITATION OF LIABILITY -
**
** IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN
** WRITING WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MODIFIES
** AND/OR DISTRIBUTES THE SOURCE CODE, BE LIABLE TO YOU FOR DAMAGES,
** INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES
** ARISING OUT OF THE USE OR INABILITY TO USE THE SOURCE CODE
** (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING RENDERED
** INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE
** OF THE SOURCE CODE TO OPERATE WITH ANY OTHER PROGRAM), EVEN IF SUCH
** HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
** DAMAGES.
**
****************************************************************************/

#ifndef PCBU_TREE_H
#define PCBU_TREE_H

#ifdef __cplusplus
extern "C"
{
#endif

#include <stddef.h>

#include <pcb/utils/linked_list.h>

/**
@ingroup pcb_utils_containers
@file
@brief
Header file with tree type definitions and tree functions
*/

/**
@ingroup pcb_utils_containers
@defgroup pcb_utils_tree Tree
@{

@brief
Hierarchical unordered tree implementations.

@details
In this tree implementation the data structure contains a pointer to the parent and a linked list of children.
Because (@ref pcb_utils_linked_list) a linked is used for building the list of children and that each tree item can
be part of this list also a list iterator member is in the stucture definition.

For convience a structure is available that defines a tree. This structure only contains a tree_item_t member.

@section pcb_utils_tree_create_items Adding data to a tree
The tree elements do not contain a pointer to the data, they are iterators. To create a tree with elements
containing data, a structure has to be defined that contains a tree_item_t\n\n
@verbatim
typedef struct _int32_tree_item {
    uint32_t childId;
    uint32_t depth;
    tree_item_t tree_item;
} int32_tree_item_t;
@endverbatim

In the above example a tree item is defined that contains two 32 bit integers (the childID and depth). Before
this can be used you have to initialize the iterator part of the structure using @ref tree_item_initialize
function. After doing that you can add an instance of this structure to any tree.

@verbatim
tree_t tree;
tree_initialize(&tree);
tree_item_t *root = tree_root(&tree);

int32_tree_item_t *item = (int32_tree_item_t *)calloc(1,sizeof(int32_tree_item_t));
tree_item_initialize(&item->tree_item);
item->childId = 1;
item->depth = 1;
tree_item_appendChild(root,&item->tree_item);
@endverbatim

To add extra items to the tree, @ref tree_item_appendChild and @ref tree_item_prependChild can be used.
To insert the children at a specific location @ref tree_item_insertChildAt can be used.

@section pcb_utils_tree_get_items Retreiving data from a tree
To retrieve data from a tree, it is needed to get the correct iterator first. This can be done by using
any of the following functions @ref tree_item_firstChild, @ref tree_item_lastChild, @ref tree_item_nextSibling,
@ref tree_item_prevSibling and @ref tree_item_childAt.
To retrieve the data from the iterator, use the macro @ref tree_item_data.

@verbatim
tree_iterator_t *some_it = tree_item_firstChild(root);
int32_tree_item_t *data_element = tree_item_data(some_it,int32_tree_item_t,tree_item);
@endverbatim

@section pcb_utils_tree_traversing Walking a tree
Some convieniance macros are provide to iterate a list: @ref tree_for_each_child and @ref
tree_for_each_child_reverse.
While using these macros do not add elements to the list or delete elements from the tree.
*/

// iterator macros
/**
@brief
Definition of a helper macro for forward iteration through the children of a tree item.

@details
\n
This <b>helper macro</b> iterates forwards through the linked list of children of a tree item\n
\n
<b>Warning</b>: don&rsquo;t <b>modify</b> the linked list during the iteration!

@param child a pointer variable to a tree item tree_item_t.
@param root_item the parent tree item (tree_item_t)
*/
#define tree_for_each_child(child, root_item) \
    for(child = tree_item_firstChild(root_item); child; child = tree_item_nextSibling(child))

/**
@brief
Definition of a helper macro for backward iteration through the children of a tree item.

@details
\n
This <b>helper macro</b> iterates backwards through the linked list of children of a tree item\n
\n
<b>Warning</b>: don&rsquo;t <b>modify</b> the linked list during the iteration!

@param child a pointer variable to a tree item.
@param root_item the parent tree item
*/
#define tree_for_each_child_reverse(child, root_item) \
    for(child = tree_item_lastChild(root_item); child; child = tree_item_prevSibling(child))

// data macro
/**
@brief
Definition of a helper macro for getting a pointer to the real data structure.

@details
\n
This <b>macro</b> calculates the address of the containing data structure.\n

@param item a pointer to the tree item for which we want to calculate the pointer to the containing structure
@param type the type to which the pointer has to be casted
@param member the name of the data member containing the list pointer
*/
#define tree_item_data(item, type, member) \
    (type *)(((void *)item) - offsetof(type,member));

/**
@brief
Tree item definition

@details
This structure defines a branch in the three.
*/
typedef struct _tree_item {
    llist_iterator_t listItem; /**< a linked list iterator, which is used to add this branch to the list of children of the parent */
    llist_t children; /**< linked list of child tree items (branches) */
} tree_item_t;

/**
@brief
Tree definition

@details
This structure defines the root of a tree. It is not needed to use this structure
to create a tree, a tree always start with a tree_item_t that has no parent.
This is a convience structure.
*/
typedef struct _tree {
    tree_item_t root; /**< The root of a tree */
} tree_t;

/**
@}
*/

// initializer functions
bool tree_initialize(tree_t *tree);
bool tree_item_initialize(tree_item_t *item);

// cleanup functions
void tree_cleanup(tree_t *list);
void tree_item_cleanup(tree_item_t *item);

// accessor functions
tree_item_t *tree_root(const tree_t *tree);

tree_item_t *tree_item_parent(const tree_item_t *item);
tree_item_t *tree_item_firstChild(tree_item_t *item);
tree_item_t *tree_item_childAt(tree_item_t *item, unsigned int index);
tree_item_t *tree_item_lastChild(tree_item_t *item);
tree_item_t *tree_item_nextSibling(const tree_item_t *item);
tree_item_t *tree_item_prevSibling(const tree_item_t *item);

// insertion functions
bool tree_item_appendChild(tree_item_t *parent, tree_item_t *child);
bool tree_item_prependChild(tree_item_t *parent, tree_item_t *child);
bool tree_item_insertChildAt(tree_item_t *parent, unsigned int index, tree_item_t *child);

// removal functions
tree_item_t *tree_item_takeChild(tree_item_t *child);
tree_item_t *tree_item_takeFirstChild(tree_item_t *parent);
tree_item_t *tree_item_takeLastChild(tree_item_t *parent);

// property functions
unsigned int tree_item_childCount(tree_item_t *parent);
bool tree_item_hasChildren(const tree_item_t *parent);

#ifdef __cplusplus
}
#endif

#endif

