/****************************************************************************
**
** Copyright (C) 2011 SoftAtHome. All rights reserved.
**
** SoftAtHome reserves all rights not expressly granted herein.
**
** - DISCLAIMER OF WARRANTY -
**
** THIS FILE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
** EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE.
**
** THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOURCE
** CODE IS WITH YOU. SHOULD THE SOURCE CODE PROVE DEFECTIVE, YOU
** ASSUME THE COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
**
** - LIMITATION OF LIABILITY -
**
** IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN
** WRITING WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MODIFIES
** AND/OR DISTRIBUTES THE SOURCE CODE, BE LIABLE TO YOU FOR DAMAGES,
** INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES
** ARISING OUT OF THE USE OR INABILITY TO USE THE SOURCE CODE
** (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING RENDERED
** INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE
** OF THE SOURCE CODE TO OPERATE WITH ANY OTHER PROGRAM), EVEN IF SUCH
** HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
** DAMAGES.
**
****************************************************************************/

#ifndef __TIMER_H__
#define __TIMER_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include <stddef.h>
#include <sys/time.h>

#include <pcb/utils/linked_list.h>

/**
@ingroup pcb_utils_socket_events
@file
@brief
Header file with timer functions
*/

/**
@ingroup pcb_utils_socket_events
@defgroup pcb_utils_timers Timers
@{

@brief
Timers can be used to do a job at regular intervals or once some time later.

@details
To have timers fully functional an event loop must be started. The timers are using the SIGALRM signal.
Multiple timers can be started, but only one timer will be the next to trigger the signal.
*/

typedef struct _pcb_timer pcb_timer_t;

/**
@brief
Timer timeout handler

@details
Definition of the timer timeout handler function
*/
typedef void (* pcb_timer_timeout_handler_t) (pcb_timer_t *timer, void *userdata);

/**
@brief
Deferred function

@details
Definition of a deferred function
*/
typedef void (* pcb_timer_deferred_function_t) (void *userdata);

/**
@brief
The timer states

@details
This enum holds all possible timer states
*/
typedef enum _pcb_timer_state {
    pcb_timer_off,       /**< Timer is not running */
    pcb_timer_started,   /**< Timer is started */
    pcb_timer_running,   /**< Timer is running */
    pcb_timer_expired,   /**< Timer has expired */
    pcb_timer_destroyed, /**< Timer is destroyed */
} pcb_timer_state_t;

/**
@brief
The timer type

@details
A type that holds all timer information.
*/
struct _pcb_timer {
    llist_iterator_t it;                    /**< Linked list iterator to put the tiomer in the global list of timers */
    struct itimerval timer;                 /**< The timer interval, if not set the timer is a single shot timer */
    pcb_timer_state_t state;                /**< The timer state */
    pcb_timer_timeout_handler_t handler;    /**< The handler function, called when a timer expires */
    void *userdata;                         /**< Some user specified data */
};

/**
@}
*/

bool pcb_timer_initializeTimers(void);
void pcb_timer_cleanupTimers(void);

void pcb_timer_calculateTimers(void);
void pcb_timer_checkTimers(void);

bool pcb_timer_enableTimers(bool enable);

pcb_timer_t *pcb_timer_create(void);
void pcb_timer_destroy(pcb_timer_t *timer);

bool pcb_timer_setInterval(pcb_timer_t *timer, unsigned int msec);
bool pcb_timer_setHandler(pcb_timer_t *timer, pcb_timer_timeout_handler_t handler);
bool pcb_timer_setUserData(pcb_timer_t *timer, void *userData);
void *pcb_timer_getUserData(pcb_timer_t *timer);

unsigned int pcb_timer_remainingTime(pcb_timer_t *timer);

bool pcb_timer_start(pcb_timer_t *timer, unsigned int timeout_msec);
bool pcb_timer_stop(pcb_timer_t *timer);

pcb_timer_state_t pcb_timer_getState(pcb_timer_t *timer);

bool pcb_timer_defer(pcb_timer_deferred_function_t fn, void *userdata, unsigned int time);

#ifdef __cplusplus
}
#endif
#endif
