/****************************************************************************
**
** Copyright (C) 2011 SoftAtHome. All rights reserved.
**
** SoftAtHome reserves all rights not expressly granted herein.
**
** - DISCLAIMER OF WARRANTY -
**
** THIS FILE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
** EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE.
**
** THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOURCE
** CODE IS WITH YOU. SHOULD THE SOURCE CODE PROVE DEFECTIVE, YOU
** ASSUME THE COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
**
** - LIMITATION OF LIABILITY -
**
** IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN
** WRITING WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MODIFIES
** AND/OR DISTRIBUTES THE SOURCE CODE, BE LIABLE TO YOU FOR DAMAGES,
** INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES
** ARISING OUT OF THE USE OR INABILITY TO USE THE SOURCE CODE
** (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING RENDERED
** INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE
** OF THE SOURCE CODE TO OPERATE WITH ANY OTHER PROGRAM), EVEN IF SUCH
** HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
** DAMAGES.
**
****************************************************************************/

#if !defined(PCB_STRING_H)
#define PCB_STRING_H

#ifdef __cplusplus
extern "C"
{
#endif

#include <time.h>
#include <stdbool.h>
#include <stdint.h>
#include <stddef.h>

/**
@ingroup pcb_utils_containers
@file
@brief
Header file with string type definitions and string functions
*/

/**
@ingroup pcb_utils_containers
@defgroup pcb_utils_string String
@{

@brief
String handling functions.

@details
This string implementation can only handle ASCII strings or UTF-8 strings.

This group of functions help in string coversion from and to other data types,
getting sub-strings, finding and replacing sub-strings, extend strings at the end as well at the beginning

The string buffer allocation is dynamic, but at initilization time, a buffer size hint can be given
*/

/**
@brief
String definition

@details
This structure defines a string.
*/
typedef struct _string {
    char *buffer; /**< the string buffer */
    size_t bufferSize; /**< size of the allocated buffer, this is not the size of the string */
} string_t;

/**
@brief
Case enumeration

@details
Enumeration of different cases. Can be used in compare functions.
*/
typedef enum _string_case {
    string_case_sensitive = 0,                 /**< Case sensitive */
    CaseSensitive = string_case_sensitive,     /**< @deprecated Use @ref string_case_sensitive*/
    string_case_insensitive,                   /**< Case insensitive */
    CaseInsensitive = string_case_insensitive, /**< @deprecated Use @ref string_case_insensitive*/
} string_case_t;

/**
@}
*/

// initializer functions
bool string_initialize(string_t *string, size_t bufferSize);

// cleanup functions
void string_cleanup(string_t *string);
void string_clear(string_t *string);

const char *string_buffer(const string_t *string);

size_t string_copy(string_t *dest, const string_t *src);

size_t string_fromBool(string_t *string, const bool data);
size_t string_fromChar(string_t *string, const char *data);
size_t string_fromInt8(string_t *string, const int8_t data);
size_t string_fromInt16(string_t *string, const int16_t data);
size_t string_fromInt32(string_t *string, const int32_t data);
size_t string_fromInt64(string_t *string, const int64_t data);
size_t string_fromUInt8(string_t *string, const uint8_t data);
size_t string_fromUInt16(string_t *string, const uint16_t data);
size_t string_fromUInt32(string_t *string, const uint32_t data);
size_t string_fromUInt64(string_t *string, const uint64_t data);
size_t string_fromDouble(string_t *string, const double data);
size_t string_fromTime(string_t *string, const struct tm *data, const char *format);

bool string_convertToBool(const string_t *string, bool *success);
static inline bool string_toBool(const string_t *string) {
    return string_convertToBool(string, NULL);
}

char *string_toChar(const string_t *string);

int8_t string_convertToInt8(const string_t *string, bool *success);
static inline int8_t string_toInt8(const string_t *string) {
    return string_convertToInt8(string, NULL);
}

int16_t string_convertToInt16(const string_t *string, bool *success);
static inline int16_t string_toInt16(const string_t *string) {
    return string_convertToInt16(string, NULL);
}

int32_t string_convertToInt32(const string_t *string, bool *success);
static inline int32_t string_toInt32(const string_t *string) {
    return string_convertToInt32(string, NULL);
}

int64_t string_convertToInt64(const string_t *string, bool *success);
static inline int64_t string_toInt64(const string_t *string) {
    return string_convertToInt64(string, NULL);
}

uint8_t string_convertToUInt8(const string_t *string, bool *success);
static inline uint8_t string_toUInt8(const string_t *string) {
    return string_convertToUInt8(string, NULL);
}

uint16_t string_convertToUInt16(const string_t *string, bool *success);
static inline uint16_t string_toUInt16(const string_t *string) {
    return string_convertToUInt16(string, NULL);
}

uint32_t string_convertToUInt32(const string_t *string, bool *success);
static inline uint32_t string_toUInt32(const string_t *string) {
    return string_convertToUInt32(string, NULL);
}

uint64_t string_convertToUInt64(const string_t *string, bool *success);
static inline uint64_t string_toUInt64(const string_t *string) {
    return string_convertToUInt64(string, NULL);
}

double string_convertToDouble(const string_t *string, bool *success);
static inline double string_toDouble(const string_t *string) {
    return string_convertToDouble(string, NULL);
}


struct tm *string_toTime(const string_t *string, const char *format);

void string_toUpper(string_t *string);
void string_toLower(string_t *string);

size_t string_appendChar(string_t *dest, const char *string);
size_t string_append(string_t *dest, const string_t *string);
size_t string_prependChar(string_t *dest, const char *string);
size_t string_prepend(string_t *dest, const string_t *string);

size_t string_appendFormat(string_t *dest, const char *format, ...);

size_t string_left(string_t *dest, const string_t *string, size_t length);
size_t string_right(string_t *dest,const string_t *string, size_t length);
size_t string_mid(string_t *dest,const string_t *string, unsigned int start, size_t length);

unsigned int string_findChar(const string_t *haystack,  unsigned int start, const char *needle, string_case_t cs);
unsigned int string_find(const string_t *haystack, unsigned int start, const string_t *needle, string_case_t cs);
bool string_containsChar(const string_t *haystack, const char *needle, string_case_t cs);
bool string_contains(const string_t *haystack, const string_t *needle, string_case_t cs);
void string_replaceChar(string_t *string, const char *search, const char *replace);
void string_replace(string_t *string, const string_t *search, const string_t *replace);

int string_compareChar(const string_t *string1, const char *string2, string_case_t cs);
int string_compare(const string_t *string1, const string_t *string2, string_case_t cs);

bool string_isNull(const string_t *string);
bool string_isEmpty(const string_t *string);
size_t string_length(const string_t *string);

void string_trimLeft(string_t *string);
void string_trimRight(string_t *string);
void string_trim(string_t *string);

bool string_isNumeric(string_t *string);

bool string_resolveEnvVar(string_t *string);
char *string_resolveEnvVarChar(const char *string);

#ifdef __cplusplus
}
#endif

#endif // PCB_STRING_H
