/****************************************************************************
**
** Copyright (C) 2011 SoftAtHome. All rights reserved.
**
** SoftAtHome reserves all rights not expressly granted herein.
**
** - DISCLAIMER OF WARRANTY -
**
** THIS FILE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
** EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO, THE IMPLIED
** WARRANTIES OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR
** PURPOSE.
**
** THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE OF THE SOURCE
** CODE IS WITH YOU. SHOULD THE SOURCE CODE PROVE DEFECTIVE, YOU
** ASSUME THE COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
**
** - LIMITATION OF LIABILITY -
**
** IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN
** WRITING WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MODIFIES
** AND/OR DISTRIBUTES THE SOURCE CODE, BE LIABLE TO YOU FOR DAMAGES,
** INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES
** ARISING OUT OF THE USE OR INABILITY TO USE THE SOURCE CODE
** (INCLUDING BUT NOT LIMITED TO LOSS OF DATA OR DATA BEING RENDERED
** INACCURATE OR LOSSES SUSTAINED BY YOU OR THIRD PARTIES OR A FAILURE
** OF THE SOURCE CODE TO OPERATE WITH ANY OTHER PROGRAM), EVEN IF SUCH
** HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH
** DAMAGES.
**
****************************************************************************/

#ifndef _PCB_CORE_NOTIFICATION_PARAMETER_H
#define _PCB_CORE_NOTIFICATION_PARAMETER_H

#ifdef __cplusplus
extern "C"
{
#endif

#include <pcb/core/notification.h>
#include <pcb/utils/variant.h>

/**
@ingroup pcb_core
@file
@brief
Header file with notification helper functions for easier access to the notification data
*/

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a bool

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a bool as @ref variant_toBool.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The converted value
*/
static inline bool notification_parameterBoolValue(const notification_t *notification, const char *name)
{
    return variant_bool(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a int8_t

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a int8_t as @ref variant_toInt8.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The converted value
*/
static inline int8_t notification_parameterInt8Value(const notification_t *notification, const char *name)
{
    return variant_int8(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a int16_t

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a int16_t as @ref variant_toInt16.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The converted value
*/
static inline int16_t notification_parameterInt16Value(const notification_t *notification, const char *name)
{
    return variant_int16(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a int32_t

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a int32_t as @ref variant_toInt32.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The converted value
*/
static inline int32_t notification_parameterInt32Value(const notification_t *notification, const char *name)
{
    return variant_int32(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a int64_t

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a int64_t as @ref variant_toInt64.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The converted value
*/
static inline int64_t notification_parameterInt64Value(const notification_t *notification, const char *name)
{
    return variant_int64(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a uint8_t

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a uint8_t as @ref variant_toUInt8.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The converted value
*/
static inline uint8_t notification_parameterUInt8Value(const notification_t *notification, const char *name)
{
    return variant_uint8(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a uint16_t

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a uint16_t as @ref variant_toUInt16.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The converted value
*/
static inline uint16_t notification_parameterUInt16Value(const notification_t *notification, const char *name)
{
    return variant_uint16(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a uint32_t

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a uint32_t as @ref variant_toUInt32.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The converted value
*/
static inline uint32_t notification_parameterUInt32Value(const notification_t *notification, const char *name)
{
    return variant_uint32(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a uint64_t

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a uint64_t as @ref variant_toUInt64.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The converted value
*/
static inline uint64_t notification_parameterUInt64Value(const notification_t *notification, const char *name)
{
    return variant_uint64(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a double

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a double as @ref variant_toDouble.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The converted value
*/
static inline double notification_parameterDoubleValue(const notification_t *notification, const char *name)
{
    return variant_double(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a char *

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a char * as @ref variant_toChar.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The converted value
*/
static inline char * notification_parameterCharValue(const notification_t *notification, const char *name)
{
    return variant_char(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a direct char * pointer to a notification parameter

@details
This function is provided for convenience. It is fetching a parameter in the notification
and returns a direct pointer to its char * value as @ref variant_da_char.\n
The return pointer MUST NOT be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The resulting char * pointer
    - NULL if the variant is invalid or not of char type
*/
static inline const char * notification_da_parameterCharValue(const notification_t *notification, const char *name)
{
    return variant_da_char(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a struct tm *

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a struct tm * as @ref variant_toDateTime.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The converted value
*/
static inline struct tm * notification_parameterDateTimeValue(const notification_t *notification, const char *name)
{
    return variant_dateTime(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a direct struct tm * pointer to a notification parameter

@details
This function is provided for convenience. It is fetching a parameter in the notification
and returns a direct pointer to its struct tm * value as @ref variant_da_dateTime.\n
The return pointer MUST NOT be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The resulting struct tm * pointer
    - NULL if the variant is invalid or not of dateTime type
*/
static inline const struct tm * notification_da_parameterDateTimeValue(const notification_t *notification, const char *name)
{
    return variant_da_dateTime(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a string_t *

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a string_t * as @ref variant_toString.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The converted value
*/
static inline string_t * notification_parameterStringValue(const notification_t *notification, const char *name)
{
    return variant_string(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a direct string_t * pointer to a notification parameter

@details
This function is provided for convenience. It is fetching a parameter in the notification
and returns a direct pointer to its string_t * value as @ref variant_da_string.\n
The return pointer MUST NOT be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The resulting string_t * pointer
    - NULL if the variant is invalid or not of string type
*/
static inline const string_t * notification_da_parameterStringValue(const notification_t *notification, const char *name)
{
    return variant_da_string(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a variant_list_t *

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a variant_list_t * as @ref variant_toList.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The converted value
*/
static inline variant_list_t * notification_parameterListValue(const notification_t *notification, const char *name)
{
    return variant_list(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a direct variant_list_t * pointer to a notification parameter

@details
This function is provided for convenience. It is fetching a parameter in the notification
and returns a direct pointer to its variant_list_t * value as @ref variant_da_list.\n
The return pointer MUST NOT be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The resulting variant_list_t * pointer
    - NULL if the variant is invalid or not of list type
*/
static inline variant_list_t * notification_da_parameterListValue(const notification_t *notification, const char *name)
{
    return variant_da_list(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a variant_map_t *

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a variant_map_t * as @ref variant_toMap.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The converted value
*/
static inline variant_map_t * notification_parameterMapValue(const notification_t *notification, const char *name)
{
    return variant_map(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a direct variant_map_t * pointer to a notification parameter

@details
This function is provided for convenience. It is fetching a parameter in the notification
and returns a direct pointer to its variant_map_t * value as @ref variant_da_map.\n
The return pointer MUST NOT be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The resulting variant_map_t * pointer
    - NULL if the variant is invalid or not of map type
*/
static inline variant_map_t * notification_da_parameterMapValue(const notification_t *notification, const char *name)
{
    return variant_da_map(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a int

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a int as @ref variant_toFd.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch

@return
    - The converted value
*/
static inline int notification_parameterFdValue(const notification_t *notification, const char *name)
{
    return variant_fd(notification_da_parameterValue(notification, name));
}

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a void *

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a void * as @ref variant_toByteArray.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch
@param size pointer to an 32 bit integer that will on return contains the size of the returned memory block

@return
    - The converted value
*/
static inline void * notification_parameterByteArrayValue(const notification_t *notification, const char *name, uint32_t *size)
{
    return variant_byteArray(notification_da_parameterValue(notification, name), size);
}

/**
@ingroup pcb_core_notification
@brief
Get a direct void * pointer to a notification parameter

@details
This function is provided for convenience. It is fetching a parameter in the notification
and returns a direct pointer to its void * value as @ref variant_da_byteArray.\n
The return pointer MUST NOT be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch
@param size pointer to an 32 bit integer that will on return contains the size of the returned memory block

@return
    - The resulting void * pointer
    - NULL if the variant is invalid or not of byteArray type
*/
static inline const void * notification_da_parameterByteArrayValue(const notification_t *notification, const char *name, uint32_t *size)
{
    return variant_da_byteArray(notification_da_parameterValue(notification, name), size);
}

/**
@ingroup pcb_core_notification
@brief
Get a notification parameter and converts it to a string_list_t *

@details
This function is provided for convenience. It is fetching a parameter in the notification
and converts it to a string_list_t * as @ref variant_toStringList.\n
The return pointer MUST be freed.\n

@param notification the notification of interest
@param name the name of the parameter to fetch
@param separator separator used to split the data into strings.

@return
    - The converted value
*/
static inline string_list_t * notification_parameterStringListValue(const notification_t *notification, const char *name, const char *separator)
{
    return variant_stringList(notification_da_parameterValue(notification, name), separator);
}

#ifdef __cplusplus
}
#endif
#endif
